<?php

namespace AminulBD\Streams\Engine;

use Symfony\Component\HttpClient\HttpClient;
use Symfony\Contracts\HttpClient\HttpClientInterface;

class Alibaba1688 implements Engine
{
    /**
     * Default headers.
     *
     * @var array|string[]
     */
    protected array $headers = [
        'Referer' => 'https://www.1688.com/',
        'User-Agent' => 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/116.0.0.0 Safari/537.36',
    ];

    /**
     * Default options.
     *
     * @var array|string[]
     */
    protected array $options = [
        'timeout' => 30,
    ];

    /**
     * Http client.
     *
     * @var \Symfony\Contracts\HttpClient\HttpClientInterface
     */
    protected HttpClientInterface $http;

    /**
     * 1688 constructor.
     *
     * @param array $options
     */
    public function __construct(array $options = [])
    {
        $this->options = array_merge($this->options, $options ?? []);
        $this->options['headers'] = array_merge($this->headers, $this->options['headers'] ?? []);
        $this->http = HttpClient::create($this->options);
    }

    /**
     * @param array $query
     *
     * @return array
     * @throws \Throwable
     */
    private function fetchFront(array $query = []): array
    {
        $defaultQuery = [
            'namespace' => 'livePCFeedQueryService',
            'widgetId' => 'livePCFeedQueryService',
            'methodName' => 'execute',
            'resourceId' => '28651',
            'type' => 2,
            'pageSize' => 20,
            'page' => 1,
        ];
        $props = [
            'namespace' => 'namespace',
            'widgetId' => 'widgetId',
            'methodName' => 'methodName',
            'resourceId' => 'categoryId',
            'type' => 'type',
            'liveId' => 'streamerId',
            'pageSize' => 'limit',
            'page' => 'page',
        ];
        $args = [];

        foreach ($props as $key => $prop) {
            if (isset($query[$prop])) {
                $args[$key] = $query[$prop];
            }
        }

        $endpoint = "https://widget.1688.com/front/getJsonComponent.json";
        $response = $this->http->request('GET', $endpoint, [
            'query' => array_merge($defaultQuery, $args),
        ]);

        if ($response->getStatusCode() === 200) {
            return (array) json_decode($response->getContent(), true);
        }

        return [];
    }

    /**
     * Map streams feed.
     *
     * @param array $feed
     *
     * @return array
     */
    private function mapStreams(array $feed): array
    {
        return array_map(function ($stream) {
            return [
                'feed_id' => $stream['feedId'] ?? null,
                'streamer_id' => $stream['id'],
                'user_id' => $stream['userId'],
                'login_id' => $stream['loginId'],
                'vendor' => '1688',
                'company' => $stream['companyName'],
                'cover' => $stream['coverImg'],
                'title' => $stream['title'],
                'offer' => $stream['offerText'],
                'items' => array_map(function ($offer) {
                    return [
                        'id' => $offer['id'],
                        'title' => $offer['title'],
                        'price' => $offer['price'],
                        'maxPrice' => $offer['maxPrice'],
                        'image' => str_replace('.220x220', '', $offer['mainPic']),
                        'url' => "https://detail.1688.com/offer/" . $offer['id'] . ".html",
                    ];
                }, $stream['offerList'] ?? []),
                'flv' => $stream['liveUrl'] ?? null,
                'm3u8' => $stream['liveUri'] ?? null,
                'timestamp' => (int) $stream['startTimeStamp'],
                'viewers' => (int) $stream['viewNum'],
            ];
        }, $feed);
    }

    /**
     * Map categories.
     *
     * @param array $data
     *
     * @return array
     */
    public function mapCategories(array $data): array
    {
        return array_map(function ($section) {
            return [
                'id' => $section['resourceId'],
                'name' => $section['name'],
            ];
        }, $data);

    }

    /**
     * Get stream feed.
     *
     * @param array $filter
     *
     * @return array
     * @throws \Throwable
     */
    public function feed(array $filter = []): array
    {
        $feedTypes = [
            'live' => '2',
            'upcoming' => '1',
            'playback' => '3',
        ];

        $type = $filter['feedType'] ?? 'live';
        if (isset($feedTypes[$type])) {
            $filter['type'] = $feedTypes[$type];
        }

        if ($type === 'category') {
            $filter['namespace'] = 'livePCMainService';
            $filter['widgetId'] = 'livePCMainService';
            $response = $this->fetchFront($filter);
            $content = $response['content'] ?? [];

            return $this->mapCategories($content['sectionList'] ?? []);
        } else {
            $response = $this->fetchFront($filter);
            $content = $response['content'] ?? [];
        }

        return [
            'data' => $this->mapStreams($content['feedList'] ?? []),
            'has_more' => $content['hasMore'] ?? false,
        ];
    }

    /**
     * Get playback streams.
     *
     * @param array $filter
     *
     * @return array
     * @throws \Throwable
     */
    public function playback(array $filter = []): array
    {
        $filter['feedType'] = 'playback';

        return $this->feed($filter);
    }

    /**
     * Get upcoming streams.
     *
     * @param array $filter
     *
     * @return array
     * @throws \Throwable
     */
    public function upcoming(array $filter = []): array
    {
        $filter['feedType'] = 'upcoming';

        return $this->feed($filter);
    }

    /**
     * Get live streams.
     *
     * @param array $filter
     *
     * @return array
     * @throws \Throwable
     */
    public function live(array $filter = []): array
    {
        $filter['feedType'] = 'live';

        return $this->feed($filter);
    }

    /**
     * Get categories.
     *
     * @param array $filter
     *
     * @return array
     * @throws \Throwable
     */
    public function categories(array $filter = []): array
    {
        $filter['feedType'] = 'category';

        return $this->feed($filter);
    }

    /**
     * Fetch history of a streamer.
     *
     * @param array $filter
     *
     * @return array
     * @throws \Throwable
     */
    public function history(array $filter = []): array
    {
        if (!isset($filter['streamerId'])) {
            throw new \Exception('The `streamerId` is required.');
        }

        $filter['namespace'] = 'tvliveserviceQueryHistoryFeedList';
        $filter['widgetId'] = 'tvliveserviceQueryHistoryFeedList';
        $filter['limit'] = 15;
        $response = $this->fetchFront($filter);
        $content = $response['content'] ?? [];
        $data = $content['res'] ?? [];
        $count = $data['count'] ?? 0;
        $page = $data['page'] ?? 1;
        $currentCount = $page * $filter['limit'];

        return [
            'data' => array_map(function ($stream) {
                return [
                    'feed_id' => $stream['feedId'] ?? null,
                    'streamer_id' => $stream['id'],
                    'user_id' => $stream['userId'],
                    'login_id' => $stream['loginId'],
                    'vendor' => '1688',
                    'company' => $stream['companyName'] ?? null,
                    'cover' => $stream['coverImg'],
                    'title' => $stream['title'],
                    'offer' => $stream['offerText'],
                    'items' => array_map(function ($offer) {
                        return [
                            'id' => $offer,
                            'title' => null,
                            'price' => null,
                            'maxPrice' => null,
                            'image' => null,
                            'url' => null,
                        ];
                    }, explode(',', $stream['offerIds'],)),
                    'flv' => $stream['liveUrl'] ?? null,
                    'm3u8' => $stream['liveUri'] ?? null,
                    'timestamp' => (int) $stream['startTimeStamp'],
                    'viewers' => (int) $stream['viewNum'],
                ];
            }, $data['model'] ?? []),
            'count' => $count,
            'has_more' => $count > $currentCount ?? true,
        ];
    }

    /**
     * Get params for request.
     *
     * @return string[]
     * @throws \Throwable
     */
    private function getParams(): array
    {
        // TODO: cache this request
        $url = 'https://h5api.m.1688.com/h5/mtop.ali.smartui.getcomponentdata/1.0/';
        $response = $this->http->request('GET', $url, [
            'query' => [
                'appKey' => '12574478',
            ],
        ]);

        $headers = $response->getHeaders();
        $data = [
            'token' => null,
            'cookies' => null,
        ];

        foreach ($headers['set-cookie'] as $cookie) {
            $cookie = explode(';', $cookie);
            $cookieKV = explode('=', $cookie[0], 2);

            if ($cookieKV[0] === '_m_h5_tk') {
                $data['token'] = explode('_', $cookieKV[1], 2)[0];
            }

            $data['cookies'] = "{$data['cookies']}$cookie[0]; ";
        }

        return $data;
    }

    /**
     * Get stream details.
     *
     * @param array $filter
     *
     * @return array
     * @throws \Throwable
     */
    public function details(array $filter = []): array
    {
        if (!isset($filter['feedId'])) {
            throw new \Exception('The `feedId` is required.');
        }

        $query = [
            'cid' => 'liveroomAggrInfo:liveroomAggrInfo',
            'methodName' => 'execute',
            'params' => json_encode([
                'feedId' => $filter['feedId'],
            ]),
        ];
        $params = $this->getParams();
        $data = [
            'token' => $params['token'],
            'time' => round(microtime(true) * 1000),
            'appKey' => '12574478',
            'data' => json_encode($query),
        ];
        $url = 'https://h5api.m.1688.com/h5/mtop.taobao.widgetservice.getjsoncomponent/1.0/';
        $response = $this->http->request('POST', $url, [
            'query' => [
                'appKey' => '12574478',
                't' => $data['time'],
                'sign' => md5(implode('&', $data)),
                'api' => 'mtop.taobao.widgetService.getJsonComponent',
                'data' => $data['data'],
            ],
            'headers' => array_merge($this->headers, [
                'Cookie' => $params['cookies'],
            ]),
        ]);

        $data = json_decode($response->getContent(), true);
        $streamInfo = $data['data']['data']['streamerInfo'] ?? [];
        $liveInfo = $data['data']['data']['liveInfo'] ?? [];

        return [
            'feed_id' => $liveInfo['feedId'] ?? null,
            'streamer_id' => $liveInfo['id'],
            'user_id' => null,
            'login_id' => $streamInfo['loginId'],
            'company' => $streamInfo['companyName'],
            'cover' => $liveInfo['coverImg'],
            'title' => $liveInfo['title'],
            'offer' => null,
            'items' => [],
            'flv' => $liveInfo['liveUrl'] ?? null,
            'm3u8' => $liveInfo['liveUri'] ?? null,
            'timestamp' => (int) $liveInfo['startTimeStamp'],
            'viewers' => (int) $liveInfo['viewNum'],
        ];
    }

    /**
     * Get offer list of a stream.
     *
     * @param array $filter
     *
     * @return array
     * @throws \Throwable
     */
    public function offers(array $filter = []): array
    {
        $required = ['streamerId', 'loginId', 'feedId'];
        foreach ($required as $key) {
            if (!isset($filter[$key])) {
                throw new \Exception("The `$key` is required.");
            }
        }

        $query = [
            'pageSize' => $filter['limit'] ?? 20,
            'currentPage' => $filter['page'] ?? 1,
            'liveId' => $filter['streamerId'],
            'streamerLoginId' => $filter['loginId'],
            'feedId' => $filter['feedId'],
        ];
        $params = $this->getParams();
        $data = [
            'token' => $params['token'],
            'time' => round(microtime(true) * 1000),
            'appKey' => '12574478',
            'data' => json_encode($query),
        ];
        $url = 'https://h5api.m.1688.com/h5/mtop.alibaba.wireless.live.querypageliveofferlist/1.0/';
        $response = $this->http->request('POST', $url, [
            'query' => [
                'appKey' => '12574478',
                't' => $data['time'],
                'sign' => md5(implode('&', $data)),
                'api' => 'mtop.alibaba.wireless.live.querypageliveofferlist',
                'data' => $data['data'],
            ],
            'headers' => array_merge($this->headers, [
                'Cookie' => $params['cookies'],
            ]),
        ]);

        $data = json_decode($response->getContent(), true);
        $offers = $data['data']['offerList'] ?? [];
        $total = $data['data']['total_count'] ?? 0;

        return [
            'data' => array_map(function ($offer) {
                return [
                    'id' => $offer['offerId'],
                    'title' => $offer['subject'],
                    'price' => $offer['minPrice'],
                    'maxPrice' => $offer['maxPrice'],
                    'image' => str_replace('.310x310', '', $offer['image']),
                    'url' => "https://detail.1688.com/offer/" . $offer['offerId'] . ".html",
                ];
            }, $offers),
            'count' => $total,
            'has_more' => $total > $query['pageSize'] * $query['currentPage'],
        ];
    }
}
