<?php

namespace AminulBD\Streams\Engine;

use Symfony\Component\HttpClient\HttpClient;
use Symfony\Contracts\HttpClient\HttpClientInterface;
use Throwable;

class Alibaba1688 implements Engine
{
    /**
     * Store response as array for cache.
     *
     * @var array
     */
    private array $cache = [];

    /**
     * Default headers.
     *
     * @var array|string[]
     */
    protected array $headers = [
        'Referer' => 'https://www.1688.com/',
        'User-Agent' => 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/116.0.0.0 Safari/537.36',
    ];

    /**
     * Default options.
     *
     * @var array|string[]
     */
    protected array $options = [
        'timeout' => 30,
    ];

    /**
     * Http client.
     *
     * @var \Symfony\Contracts\HttpClient\HttpClientInterface
     */
    protected HttpClientInterface $http;

    /**
     * 1688 constructor.
     *
     * @param array $options
     */
    public function __construct(array $options = [])
    {
        $this->options = array_merge($this->options, $options ?? []);
        $this->options['headers'] = array_merge($this->headers, $this->options['headers'] ?? []);
        $this->http = HttpClient::create($this->options);
    }

    /**
     * @param array $query
     */
    private function fetch(array $query = []): void
    {
        $url = "https://widget.1688.com/front/getJsonComponent.json";
        $defaultQuery = [
            'namespace' => 'livePCMainService',
            'widgetId' => 'livePCMainService',
            'methodName' => 'execute',
        ];

        try {
            $response = $this->http->request('GET', $url, [
                'query' => array_merge($defaultQuery, $query)
            ]);

            if ($response->getStatusCode() === 200) {
                $this->cache = json_decode($response->getContent(), true);
            }
        } catch (Throwable $exception) {
            $this->cache = [];
        }
    }

    /**
     * Get streams and format.
     *
     * @param string $key
     *
     * @return array
     */
    private function getStreams(string $key): array
    {
        if (empty($this->cache)) {
            $this->fetch();
        }

        try {
            $data = $this->cache['content'][$key];
        } catch (Throwable $e) {
            return [];
        }

        return array_map(function ($stream) {
            return [
                'id' => $stream['feedId'] ?? null,
                'user_id' => $stream['userId'],
                'user' => $stream['loginId'],
                'vendor' => '1688',
                'company' => $stream['companyName'],
                'cover' => $stream['coverImg'],
                'title' => $stream['title'],
                'offer' => $stream['offerText'],
                'items' => array_map(function ($offer) {
                    return [
                        'id' => $offer['id'],
                        'title' => $offer['title'],
                        'price' => $offer['price'],
                        'maxPrice' => $offer['maxPrice'],
                        'image' => str_replace('.220x220', '', $offer['mainPic']),
                        'url' => "https://detail.1688.com/offer/" . $offer['id'] . ".html",
                    ];
                }, $stream['offerList'] ?? []),
                'flv' => $stream['liveUrl'] ?? null,
                'm3u8' => $stream['liveUri'] ?? null,
                'timestamp' => (int) $stream['startTimeStamp'],
                'viewers' => (int) $stream['viewNum'],
            ];
        }, $data);
    }

    /**
     * Get streams by category.
     *
     * @param string $key
     *
     * @return $this
     */
    public function byCategory(string $key): self
    {
        $this->fetch(['resourceId' => $key]);

        return $this;
    }

    /**
     * Get playback streams.
     *
     * @return array
     *
     */
    public function playback(): array
    {
        return $this->getStreams('replayFeedList');
    }

    /**
     * Get upcoming streams.
     *
     * @return array
     */
    public function upcoming(): array
    {
        return $this->getStreams('preFeedList');
    }

    /**
     * Get live streams.
     *
     * @return array
     */
    public function live(): array
    {
        return $this->getStreams('liveFeedList');
    }

    /**
     * Get categories.
     *
     * @return array
     */
    public function categories(): array
    {
        if (empty($this->cache)) {
            $this->fetch();
        }

        try {
            $data = $this->cache['content']['sectionList'];

            return array_map(function ($section) {
                return [
                    'id' => $section['resourceId'],
                    'name' => $section['name'],
                ];
            }, $data ?? []);
        } catch (Throwable $e) {
            return [];
        }
    }
}
