<?php

namespace AminulBD\Streams\Engine;

use Illuminate\Support\Facades\Http;

class Alibaba1688 implements Engine
{
    /**
     * Store response as array for cache.
     *
     * @var array
     */
    private array $cache = [];

    /**
     * Fetch data from 1688.
     *
     * @param array $query
     *
     * @return void
     */
    private function fetch(array $query = []): void
    {
        $url = "https://widget.1688.com/front/getJsonComponent.json";
        $defaultQuery = [
            'namespace' => 'livePCMainService',
            'widgetId' => 'livePCMainService',
            'methodName' => 'execute',
        ];
        $headers = [
            'Referer' => 'https://www.1688.com/',
            'User-Agent' => 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/116.0.0.0 Safari/537.36',
        ];

        $response = Http::withOptions(['timeout' => 60])
            ->withHeaders(array_merge($headers))
            ->get($url, array_merge($defaultQuery, $query));

        if ($response->ok()) {
            $this->cache = $response->json();
        }
    }

    /**
     * Get streams and format.
     *
     * @param string $key
     *
     * @return array
     */
    private function getStreams(string $key): array
    {
        if (empty($this->cache)) {
            $this->fetch();
        }

        $data = data_get($this->cache, $key);

        return array_map(function ($stream) {
            return [
                'id' => $stream['feedId'] ?? null,
                'user_id' => $stream['userId'],
                'user' => $stream['loginId'],
                'vendor' => '1688',
                'company' => $stream['companyName'],
                'cover' => $stream['coverImg'],
                'title' => $stream['title'],
                'offer' => $stream['offerText'],
                'items' => array_map(function ($offer) {
                    return [
                        'id' => $offer['id'],
                        'title' => $offer['title'],
                        'price' => $offer['price'],
                        'maxPrice' => $offer['maxPrice'],
                        'image' => str_replace('.220x220', '', $offer['mainPic']),
                        'url' => "https://detail.1688.com/offer/" . $offer['id'] . ".html",
                    ];
                }, $stream['offerList'] ?? []),
                'flv' => $stream['liveUrl'] ?? null,
                'm3u8' => $stream['liveUri'] ?? null,
                'timestamp' => (int) $stream['startTimeStamp'],
                'viewers' => (int) $stream['viewNum'],
            ];
        }, $data);
    }

    /**
     * Get streams by category.
     *
     * @param string $key
     *
     * @return $this
     */
    public function byCategory(string $key): self
    {
        $this->fetch(['resourceId' => $key]);

        return $this;
    }

    /**
     * Get playback streams.
     *
     * @return array
     */
    public function playback(): array
    {
        return $this->getStreams('content.replayFeedList');
    }

    /**
     * Get upcoming streams.
     *
     * @return array
     */
    public function upcoming(): array
    {
        return $this->getStreams('content.preFeedList');
    }

    /**
     * Get live streams.
     *
     * @return array
     */
    public function live(): array
    {
        return $this->getStreams('content.liveFeedList');
    }

    /**
     * Get categories.
     *
     * @return array
     */
    public function categories(): array
    {
        if (empty($this->cache)) {
            $this->fetch();
        }

        $data = data_get($this->cache, 'content.sectionList');

        return array_map(function ($section) {
            return [
                'id' => $section['resourceId'],
                'name' => $section['name'],
            ];
        }, $data ?? []);
    }
}
