<?php

namespace AminulBD\Taobao\Laravel;

use Illuminate\Http\Client\Response;
use Illuminate\Support\Facades\Http;

class Client
{
    /**
     * @var array
     */
    private $config;

    /**
     * @param array $config
     */
    public function __construct(array $config)
    {
        $requiredFields = [
            'app_key', 'app_secret',
        ];

        foreach ($requiredFields as $field) {
            if (!isset($config[$field])) {
                throw new \InvalidArgumentException("Missing required config field: $field");
            }
        }

        $baseUrl = trim($config['base_url'] ?? 'https://api.taobao.global', '/');
        if (!filter_var($baseUrl, FILTER_VALIDATE_URL)) {
            throw new \InvalidArgumentException("Invalid base URL: $baseUrl");
        }

        $this->config = $config;
    }

    /**
     * @param string $api
     * @param array $params
     * @return array
     */
    private function prepare(string $api, array $params): array
    {
        $params['app_key'] = $this->config['app_key'];
        $params['sign_method'] = 'sha256';
        $params['timestamp'] = time() * 1000;
        $params['method'] = $api;

        ksort($params);
        $stringToSign = $api;

        foreach ($params as $key => $value) {
            $stringToSign .= "$key$value";
        }

        $sign = strtoupper(hash_hmac('sha256', $stringToSign, $this->config['app_secret']));

        return array_merge($params, [
            'sign' => $sign,
        ]);
    }

    /**
     * Generate the auth URL.
     *
     * @param string|null $callbackUrl
     * @return string
     */
    public function getAuthorizeUrl(string $callbackUrl = '', string $state = ''): string
    {
        $query = [
            'response_type' => 'code',
            'force_auth' => 'true',
            'redirect_uri' => !empty($callbackUrl) ? $callbackUrl : $this->config['redirect_uri'],
            'client_id' => $this->config['app_key'],
        ];

        if (!empty($state)) {
            $query['state'] = $state;
        }

        return $this->config['base_url'] . '/oauth/authorize?' . http_build_query($query);
    }

    /**
     * Get access token by Code
     *
     * @param string $code
     *
     * @return \Illuminate\Http\Client\Response
     * @throws \Illuminate\Http\Client\ConnectionException
     */
    public function getAccessToken(string $code): Response
    {
        return $this->request('/auth/token/create', [
            'code' => $code,
        ]);
    }

    /**
     * Refresh the access token using the refresh token.
     *
     * @param string $token
     *
     * @return \Illuminate\Http\Client\Response
     * @throws \Illuminate\Http\Client\ConnectionException
     */
    public function refreshAccessToken(string $token): Response
    {
        return $this->request('/auth/token/refresh', [
            'refresh_token' => $token,
        ]);
    }

    /**
     * Make a request to the AliExpress API.
     *
     * @param string $api
     * @param array $params
     *
     * @return \Illuminate\Http\Client\Response
     * @throws \Illuminate\Http\Client\ConnectionException
     */
    public function request(string $api, array $params = []): Response
    {
        $data = $this->prepare($api, $params);
        $url = $this->config['base_url'] . '/rest';
        $res = Http::baseUrl($url)->asForm()->post($api, $data);

        if (!$res->ok()) {
            throw new \RuntimeException('Request failed with status code: ' . $res->status());
        }

        return $res;
    }
}
