<?php

namespace MoveOn\Cartay\Drivers;

use Exception;
use GuzzleHttp\Client;
use GuzzleHttp\Cookie\FileCookieJar;
use GuzzleHttp\Cookie\SetCookie;
use GuzzleHttp\Exception\GuzzleException;
use MoveOn\Cartay\Cart;
use MoveOn\Cartay\CartItem;
use MoveOn\Cartay\Contracts\Account;
use MoveOn\Cartay\Exceptions\CartayException;
use MoveOn\Cartay\OrderItem;
use MoveOn\Cartay\Orders;
use MoveOn\Cartay\Tracking;

class Taobao extends Driver
{
    private FileCookieJar $cookieJar;
    /**
     * @var \GuzzleHttp\Client
     */
    private Client $client;

    /**
     * @var string
     */
    private string $userAgent;

    /**
     * Taobao driver constructor.
     *
     * @param \MoveOn\Cartay\Contracts\Account $account
     */
    public function __construct(Account $account)
    {
        parent::__construct($account);

        $this->userAgent = 'Mozilla/5.0 (Linux; Android 8.0; Pixel 2 Build/OPD3.170816.012) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/92.0.4515.159 Mobile Safari/537.36';
        $this->cookieJar = new FileCookieJar($this->cookiesFile, true);
        $this->client    = $this->client();
    }

    /**
     * Parse product ID from link.
     *
     * @param $link
     *
     * @return string
     */
    public static function parseIdFromLink($link): ?string
    {
        $parseUrl = parse_url($link, PHP_URL_QUERY);
        parse_str($parseUrl, $parseQuery);

        return $parseQuery['id'] ?? null;
    }

    /**
     * The client.
     *
     * @return \GuzzleHttp\Client
     */
    private function client(): Client
    {
        $cookies = $this->cookies;
        $headers = $this->headers;

        if ($this->cookieJar->count() === 0 && count($cookies) > 0) {
            foreach ($this->cookies as $name => $value) {
                $this->cookieJar->setCookie(new SetCookie([
                    'Name'   => $name,
                    'Value'  => $value,
                    'Domain' => '.taobao.com',
                ]));
            }
        }

        if (empty($headers['Referer']) || empty($headers['referer'])) {
            $headers['Referer'] = 'https://item.taobao.com';
        }

        if (empty($headers['User-Agent']) || empty($headers['user-agent'])) {
            $headers['User-Agent'] = $this->userAgent;
        }

        return new Client([
            'cookies'     => $this->cookieJar,
            'headers'     => $headers,
            'http_errors' => false,
            'timeout'     => 30,
        ]);
    }

    /**
     * @inheritDoc
     */
    public function cart(): Cart
    {
        try {
            $data = [
                'exParams'  => json_encode([
                    'mergeCombo' => 'true',
                    'version'    => '1.0.0',
                    'globalSell' => '1',
                ], JSON_UNESCAPED_SLASHES),
                'isPage'    => "false",
                'extStatus' => "0",
            ];

            $signData = [
                'token'  => $this->getTokenForSign(),
                'time'   => round(microtime(true) * 1000),
                'appKey' => '12574478',
                'data'   => json_encode($data),
            ];
            $cartUrl  = 'https://acs.m.taobao.com/h5/mtop.trade.querybag/5.0/';
            $cartPage = $this->client->get($cartUrl, [
                'headers' => [
                    'Content-Type' => 'application/x-www-form-urlencoded',
                    'Accept'       => 'application/json',
                    'Referer'      => 'https://h5.m.taobao.com/',
                ],
                'query'   => [
                    'jsv'          => '2.3.26',
                    'appKey'       => $signData['appKey'],
                    't'            => $signData['time'],
                    'sign'         => $this->generateSign($signData),
                    'api'          => 'mtop.trade.queryBag',
                    'v'            => '5.0',
                    'isSec'        => '0',
                    'ecode'        => '1',
                    'AntiFlood'    => 'true',
                    'AntiCreep'    => 'true',
                    'H5Request'    => 'true',
                    'LoginRequest' => 'true',
                    'type'         => 'originaljson',
                    'dataType'     => 'jsonp',
                    'data'         => $signData['data'],
                ],
            ]);

            if ($cartPage->getStatusCode() !== 200) {
                $this->cart->setStatus(
                    'error',
                    'Invalid response from the cart server. HTTP Status Code: ' . $cartPage->getStatusCode()
                );

                return $this->cart;
            }

            $data = json_decode($cartPage->getBody()->getContents(), true);
            $data = $data['data'] ?? [];
            $data = $data['data'] ?? [];

            $products    = array_filter((array) $data, fn($item) => isset($item['tag']) && $item['tag'] === 'itemv2');
            $shops       = array_filter((array) $data, fn($item) => isset($item['tag']) && $item['tag'] === 'shopv2');
            $settlements = array_map(fn($item) => $item['fields']['settlement'], $products);
            $freights    = $this->fetchFreights($settlements);

            foreach ($products as $product) {
                $fields   = $product['fields'];
                $shop     = $shops['shopv2_' . $fields['bundleId']] ?? [];
                $shop     = $shop['fields'] ?? [];
                $cartItem = new CartItem([
                    'seller' => [
                        'id'          => $shop['shopId'],
                        'memberId'    => $shop['sellerId'],
                        'companyName' => $shop['title'],
                        'companyUrl'  => "https://store.taobao.com/shop/view_shop.htm?user_number_id={$shop['sellerId']}",
                    ],
                ]);

                $cartItem->id          = $fields['cartId'] ?? null;
                $cartItem->sellerId    = $cartItem->seller['id'];
                $cartItem->productId   = $fields['itemId'] ?? null;
                $cartItem->quantityMin = $fields['quantity']['min'] ?? null;
                $cartItem->quantityMax = $fields['quantity']['max'] ?? null;
                $cartItem->title       = $fields['title'] ?? null;
                $cartItem->link        = "https://item.taobao.com/item.htm?id=$cartItem->productId";
                $imageUrl              = str_replace('.jpg_sum.jpg', '.jpg', $fields['pic']);
                $cartItem->image       = $imageUrl;
                $cartItem->thumb       = $imageUrl ? str_replace('.jpg', '.jpg_50x50.jpg', $imageUrl) : null;

                $attrs    = isset($fields['sku']['title']) ? array_map(
                    fn($item) => ['key' => $item, 'value' => $item],
                    explode(';', $fields['sku']['title'])
                ) : [];
                $quantity = $fields['quantity']['quantity'] ?? 1;
                $skuId    = $fields['sku']['skuId'] ?? null;

                $cartItem->sku = [
                    'id'          => $cartItem->id ?? null,
                    'quantity'    => $quantity,
                    'quantityMin' => $cartItem->quantityMin ?? null,
                    'quantityMax' => $cartItem->quantityMax ?? null,
                    'skuId'       => $skuId !== '0' ? $skuId : null,
                    'image'       => $imageUrl,
                    'attrs'       => $attrs,
                    'skuName'     => null,
                    'unit'        => null, // piece, lot, box, kg
                    'price'       => ($fields['pay']['total'] / $quantity) / 100,
                ];

                $cartItem->current_freight    = $freights[$cartItem->id] ?? [];
                $cartItem->available_freights = [];

                $this->cart->add($cartItem);
            }

            $this->cart->setStatus(
                'success',
                'Cart items are fetched.'
            );
        } catch (GuzzleException $exception) {
            $this->cart->setStatus('error', $exception->getMessage());
        }

        $this->cart->limit = 50;

        return $this->cart;
    }

    /**
     * @inheritDoc
     */
    public function add(string $link, array $skus): bool
    {
        $productId = self::parseIdFromLink($link);
        if (empty($productId)) {
            throw new CartayException('Unable to parse product id from the link.');
        }
        try {
            $added = 0;
            foreach ($skus as $sku) {
                $data = [
                    'itemId'   => $productId,
                    'quantity' => $sku['qty'],
                ];

                if (!empty($sku['id'])) {
                    $data['skuId'] = $sku['id'];
                }

                $signData = [
                    'token'  => $this->getTokenForSign(),
                    'time'   => round(microtime(true) * 1000),
                    'appKey' => '12574478',
                    'data'   => json_encode($data),
                ];
                $cartUrl  = 'https://h5api.m.taobao.com/h5/mtop.trade.addbag/3.1/';
                $request  = $this->client->post($cartUrl, [
                    'headers'     => [
                        'Content-Type' => 'application/x-www-form-urlencoded',
                        'Accept'       => 'application/json',
                        'Referer'      => 'https://m.intl.taobao.com/',
                    ],
                    'query'       => [
                        'jsv'      => '2.6.0',
                        'appKey'   => $signData['appKey'],
                        't'        => $signData['time'],
                        'sign'     => $this->generateSign($signData),
                        'api'      => 'mtop.trade.addBag',
                        'v'        => '3.1',
                        'type'     => 'originaljson',
                        'ecode'    => '1',
                        'dataType' => 'jsonp',
                    ],
                    'form_params' => [
                        'data' => $signData['data'],
                    ],
                ]);

                if ($request->getStatusCode() !== 200) {
                    throw new CartayException('Invalid response from the cart server.');
                }

                // TODO: Determine the cart is updated, added or failed.

                $added++;
            }

            return $added > 0;
        } catch (GuzzleException $exception) {
            throw new CartayException($exception->getMessage());
        }
    }

    /**
     * @inheritDoc
     */
    public function update(string $itemId, array $params): bool
    {
        try {
            $data = [
                'cartId'   => $itemId,
                'quantity' => $params['quantity'],
            ];

            $dataKey   = 'itemv2_' . $itemId;
            $actionKey = $params['action'] ?? 'update';

            $signData = [
                'token'  => $this->getTokenForSign(),
                'time'   => round(microtime(true) * 1000),
                'appKey' => '12574478',
                'data'   => json_encode([
                    'p' => json_encode([
                        'data'    => [
                            $dataKey => [
                                'fields' => $data,
                            ],
                        ],
                        'operate' => [
                            $actionKey => [$dataKey],
                        ],
                    ]),
                ]),
            ];
            $cartUrl  = 'https://acs.m.taobao.com/h5/mtop.trade.updatebag/4.0/';
            $request  = $this->client->post($cartUrl, [
                'headers'     => [
                    'Content-Type' => 'application/x-www-form-urlencoded',
                    'Accept'       => 'application/json',
                    'Referer'      => 'https://h5.m.taobao.com/',
                ],
                'query'       => [
                    'jsv'          => '2.3.26',
                    'appKey'       => $signData['appKey'],
                    't'            => $signData['time'],
                    'sign'         => $this->generateSign($signData),
                    'api'          => 'mtop.trade.updateBag',
                    'v'            => '4.0',
                    'type'         => 'originaljson',
                    'isSec'        => '0',
                    'ecode'        => '1',
                    'dataType'     => 'jsonp',
                    'AntiFlood'    => 'true',
                    'AntiCreep'    => 'true',
                    'H5Request'    => 'true',
                    'LoginRequest' => 'true',
                ],
                'form_params' => [
                    'data' => $signData['data'],
                ],
            ]);

            if ($request->getStatusCode() !== 200) {
                throw new CartayException('Invalid response from the cart server.');
            }

            // TODO: Determine the cart is updated, added or failed.

            return true;
        } catch (GuzzleException $exception) {
            throw new CartayException($exception->getMessage());
        }
    }

    /**
     * @inheritDoc
     */
    public function remove(string $itemId): bool
    {
        return $this->update($itemId, ['quantity' => 1, 'action' => 'delete']);
    }

    /**
     * @inheritDoc
     */
    public function orders(int $page = 1, array $filters = []): Orders
    {
        try {
            $request = $this->client->post('https://buyertrade.taobao.com/trade/itemlist/asyncBought.htm', [
                'headers'     => [
                    'Referer' => 'https://buyertrade.taobao.com/trade/itemlist/list_bought_items.htm',
                ],
                'query'       => [
                    'action'                => 'itemlist/BoughtQueryAction',
                    'event_submit_do_query' => '1',
                    '_input_charset'        => 'utf8',
                ],
                'form_params' => [
                    'buyerNick'        => $filters['buyerNick'] ?? '',
                    'dateBegin'        => $filters['dateBegin'] ?? '',
                    'dateEnd'          => $filters['dateEnd'] ?? '',
                    'lastStartRow'     => $filters['lastStartRow'] ?? '',
                    'logisticsService' => $filters['logisticsService'] ?? '',
                    'options'          => $filters['options'] ?? '',
                    'orderStatus'      => $filters['orderStatus'] ?? '',
                    'pageNum'          => $page,
                    'pageSize'         => '15',
                    'queryBizType'     => $filters['queryBizType'] ?? '',
                    'queryOrder'       => $filters['queryOrder'] ?? '',
                    'rateStatus'       => $filters['rateStatus'] ?? '',
                    'refund'           => $filters['refund'] ?? '',
                    'sellerNick'       => $filters['sellerNick'] ?? '',
                    'prePageNo'        => $page === 1 ? 2 : $page - 1,
                ],
            ]);

            if ($request->getStatusCode() !== 200) {
                $this->orders->setStatus('error', 'Invalid response from the orders server. Code: ' . $request->getStatusCode());

                return $this->orders;
            }

            $data = mb_convert_encoding($request->getBody()->getContents(), 'UTF-8', 'GBK');
            $data = json_decode(trim($data), true);
            if (!is_array($data) || !isset($data['mainOrders'])) {
                $this->orders->setStatus('error', 'The order list is empty.');

                return $this->orders;
            }

            foreach ($data['mainOrders'] as $order) {
                $skus = [];

                foreach ($order['subOrders'] as $item) {
                    if (isset($item['itemInfo']['skuText']) && !is_array($item['itemInfo']['skuText'])) {
                        continue;
                    }

                    $attrs = array_map(function ($attr) {
                        return [
                            'key'   => $attr['name'] ?? null,
                            'value' => $attr['value'] ?? null,
                        ];
                    }, $item['itemInfo']['skuText'] ?? []);

                    if (isset($item['id'])) {
                        $skus[] = [
                            'id'         => $item['id'],
                            'product_id' => $item['itemInfo']['id'],
                            'title'      => $item['itemInfo']['title'],
                            'link'       => $item['itemInfo']['itemUrl'],
                            'image'      => $item['itemInfo']['pic'],
                            'price'      => $item['priceInfo']['realTotal'],
                            'price_real' => $item['priceInfo']['original'] ?? null,
                            'quantity'   => (int) $item['quantity'],
                            'sku_id'     => $item['itemInfo']['skuId'] !== -1 ? $item['itemInfo']['skuId'] : null,
                            'attributes' => $attrs,
                            'status'     => $order['statusInfo']['text'],
                        ];
                    }
                }

                $attributes = [
                    'id'              => $order['id'],
                    'total'           => $order['payInfo']['actualFee'],
                    'real_total'      => null,
                    'shipping_charge' => null,
                    'buyer_id'        => null,
                    'buyer_member_id' => null,
                    'buyer_account'   => null,
                    'created_at'      => $order['orderInfo']['createTime'],
                    'status'          => $order['statusInfo']['text'],
                    'refund_status'   => null,
                    'seller'          => [
                        'id'   => $order['seller']['id'],
                        'name' => $order['seller']['shopName'],
                        'nick' => $order['seller']['nick'],
                        'url'  => $order['seller']['shopUrl'],
                    ],
                    'skus'            => $skus,
                ];
                $this->orders->add(new OrderItem($attributes));
            }

            $this->orders->setStatus('success', 'Order list has been fetched.');

            $this->orders->currentPage = $page;
            $this->orders->pageCount   = isset($data['page']) ? $data['page']['totalPage'] : 0;
        } catch (GuzzleException $exception) {
        }

        return $this->orders;
    }

    /**
     * @inheritDoc
     */
    public function tracking(string $orderId, array $params = []): Tracking
    {
        $trackingModel = new Tracking;
        try {
            $request = $this->client->get('https://buyertrade.taobao.com/trade/json/transit_step.do', [
                'headers' => [
                    'Referer' => 'https://buyertrade.taobao.com/trade/itemlist/list_bought_items.htm',
                ],
                'query'   => [
                    'bizOrderId' => $orderId,
                ],
            ]);

            if ($request->getStatusCode() !== 200) {
                return $trackingModel;
            }

            $data         = mb_convert_encoding($request->getBody()->getContents(), 'UTF-8', 'GBK');
            $trackingData = json_decode($data, true);
            if (!is_array($trackingData) || !isset($trackingData['expressId'])) {
                $trackingModel->setStatus('error', 'Unable to parse the tracking code.');

                return $trackingModel;
            }

            $trackingModel->setCode($trackingData['expressId']);
            $trackingModel->setStatus('success', 'Tracking code has been parsed.');
            $trackingModel->setAttributes((array) $trackingData);

            return $trackingModel;
        } catch (GuzzleException $exception) {
            $trackingModel->setStatus('error', 'Guzzle Exception: ' . $exception->getMessage());

            return $trackingModel;
        }
    }

    /**
     * Get token for sign from cookies.
     *
     * @return string
     */
    private function getTokenForSign(): string
    {
        $token = $this->cookieJar->getCookieByName('_m_h5_tk');
        $token = $token ? $token->getValue() : '';

        return explode('_', $token)[0] ?? '';
    }

    /**
     * Generate sign for request.
     *
     * @param array $attributes
     *
     * @return string
     */
    private function generateSign(array $attributes): string
    {
        return md5(implode('&', $attributes));
    }

    /**
     * Fetch freights.
     *
     * @param array $keys
     *
     * @return array
     */
    private function fetchFreights(array $keys): array
    {
        try {
            $data = [
                'buyNow'   => 'false',
                'spm'      => null,
                'buyParam' => implode(',', $keys),
                'exParams' => json_encode([
                    'tradeProtocolFeatures' => '5',
                    'userAgent'             => $this->userAgent,
                ], JSON_UNESCAPED_SLASHES),
            ];

            if (!empty($sku['id'])) {
                $data['skuId'] = $sku['id'];
            }

            $signData = [
                'token'  => $this->getTokenForSign(),
                'time'   => round(microtime(true) * 1000),
                'appKey' => '12574478',
                'data'   => json_encode($data),
            ];
            $cartUrl  = 'https://h5api.m.taobao.com/h5/mtop.trade.order.build.h5/4.0/';
            $request  = $this->client->post($cartUrl, [
                'headers'     => [
                    'Content-Type' => 'application/x-www-form-urlencoded',
                    'Accept'       => 'application/json',
                    'Referer'      => 'https://main.m.taobao.com/',
                ],
                'query'       => [
                    'jsv'       => '2.6.1',
                    'appKey'    => $signData['appKey'],
                    't'         => $signData['time'],
                    'sign'      => $this->generateSign($signData),
                    'api'       => 'mtop.trade.order.build.h5',
                    'v'         => '4.0',
                    'type'      => 'originaljson',
                    'ttid'      => '#t#ip##_h5_2019',
                    'isSec'     => '1',
                    'ecode'     => '1',
                    'AntiFlood' => 'true',
                    'AntiCreep' => 'true',
                    'H5Request' => 'true',
                    'dataType'  => 'jsonp',
                ],
                'form_params' => [
                    'data' => $signData['data'],
                ],
            ]);

            if ($request->getStatusCode() !== 200) {
                return [];
            }

            $data = json_decode($request->getBody()->getContents(), true);
            if (!is_array($data)) {
                return [];
            }

            $orders   = array_filter(
                $data['data']['hierarchy']['structure'],
                fn($value, $key) => strpos($key, 'order_') !== false,
                ARRAY_FILTER_USE_BOTH
            );
            $freights = [];

            if (count($orders) > 0) {
                foreach ($orders as $values) {
                    $parcel = array_values(array_filter($values, fn($value) => strpos($value, 'deliveryMethod_') !== false));
                    $parcel = $data['data']['data'][$parcel[0]];

                    preg_match('/[0-9.]+/', $parcel['fields']['value'], $freightCosts);

                    $items = array_filter($values, fn($value) => strpos($value, 'item_') !== false);

                    foreach ($items as $item) {
                        $valueData         = $data['data']['data'][$item];
                        $itemId            = (string) $valueData['hidden']['extensionMap']['cartId'];
                        $freights[$itemId] = [
                            'id'        => 'TAOBAO_DEFAULT',
                            'name'      => $parcel['fields']['desc'] ?? null,
                            'cost'      => $freightCosts[0] ?? '0.00',
                            'trackable' => null,
                        ];
                    }
                }
            }

            return $freights;
        } catch (GuzzleException | Exception $exception) {
            return [];
        }
    }
}