<?php

namespace MoveOn\Cartay\Drivers;

use GuzzleHttp\Client;
use GuzzleHttp\Cookie\FileCookieJar;
use GuzzleHttp\Cookie\SetCookie;
use GuzzleHttp\Exception\GuzzleException;
use Illuminate\Support\Collection;
use MoveOn\Cartay\Cart;
use MoveOn\Cartay\CartItem;
use MoveOn\Cartay\Exceptions\CartayException;
use MoveOn\Cartay\OrderItem;
use MoveOn\Cartay\Orders;
use MoveOn\Cartay\Tracking;
use Symfony\Component\DomCrawler\Crawler;

class Alibaba1688 extends Driver
{
    /**
     * @var \Symfony\Component\DomCrawler\Crawler|null
     */
    private ?Crawler $crawler = null;

    /**
     * The client.
     *
     * @return \GuzzleHttp\Client
     */
    private function client(): Client
    {
        $jar = new FileCookieJar($this->cookiesFile, true);

        if ($jar->count() === 0) {
            foreach ($this->cookies as $name => $value) {
                $jar->setCookie(new SetCookie([
                    'Name'   => $name,
                    'Value'  => $value,
                    'Domain' => '.1688.com',
                ]));
            }
        }

        return new Client([
            'cookies'     => $jar,
            'headers'     => $this->headers,
            'http_errors' => false,
            'timeout'     => 30,
        ]);
    }

    /**
     * Parse product id from the link.
     *
     * @param string $link
     *
     * @return mixed|null
     */
    public static function parseIdFromLink(string $link)
    {
        $pid = null;

        preg_match('/(?<=\/offer\/)(.*)(?=.html)/', $link, $matches);
        if (is_array($matches) && count($matches) > 0) {
            $pid = $matches[0];
        }

        return $pid;
    }

    /**
     * @inheritDoc
     */
    public function cart(): Cart
    {
        try {
            $url     = 'https://cart.1688.com/cart.htm';
            $request = $this->client()->get($url);

            if ($request->getStatusCode() !== 200) {
                $this->cart->setStatus(
                    'error',
                    'Invalid response from the cart server. HTTP Status Code: ' . $request->getStatusCode()
                );
            } else {
                $this->crawler = new Crawler($request->getBody()->getContents());

                $this->parseCartItems();
            }
        } catch (GuzzleException $exception) {
            $this->cart->setStatus(
                'error',
                $exception->getMessage()
            );
        }

        return $this->cart;
    }

    /**
     * @inheritDoc
     */
    public function add(string $link, array $skus): bool
    {
        $productId = self::parseIdFromLink($link);
        if (empty($productId)) {
            throw new CartayException('Unable to parse product id from the link.');
        }

        $csrfToken = $this->getCartCSRFToken();

        $postData = [
            'type'            => 'offer',
            'cargoIdentity'   => $productId,
            'returnType'      => 'url',
            'needTotalPrice'  => false,
            'promotionSwitch' => false,
            '_csrf_token'     => $csrfToken,
            't'               => round(microtime(true) * 1000),
            'purchaseType'    => '',
            'ext'             => json_encode([
                'sceneCode' => '',
            ]),
        ];

        $idColumn = array_column($skus, 'id');
        $hasIdOne = in_array("1", $idColumn);
        if (count($skus) === 1 && $hasIdOne) {
            $postData['quantity'] = $skus[0]['qty'];
        } else {
            $items = array_filter(array_map(function ($item) {
                if (empty($item['id'])) {
                    return null;
                }
                return [
                    'specId' => $item['id'],
                    'amount' => $item['qty'] ?? 1,
                ];
            }, $skus));

            if (count($items) >= 1) {
                $postData['specData'] = json_encode($items);
            }
        }

        // TODO: Check if the item already added on the cart.
        $body = http_build_query($postData);

        try {
            $cartActionUrl = 'https://cart.1688.com/ajax/safe/add_to_cart_list_new.jsx';
            $proxyUrl      = 'https://cart.1688.com/message_proxy_server';
            $fakeTime      = round(microtime(true) * 1000);
            $proxyReq      = $this->client()->post($proxyUrl, [
                'headers' => [
                    'Content-Type' => 'application/x-www-form-urlencoded; charset=UTF-8',
                ],
                'body'    => http_build_query([
                    'data'     => $body,
                    'type'     => 'postMessage',
                    'url'      => $cartActionUrl,
                    'callback' => 'gen_id_' . $fakeTime . '_' . rand(179000000000000, 179999999999999),
                ]),
            ]);

            if ($proxyReq->getStatusCode() !== 200) {
                throw new CartayException('Pre validation has been failed.');
            }

            $request = $this->client()->post($cartActionUrl, [
                'headers' => [
                    'Content-Type' => 'application/x-www-form-urlencoded; charset=UTF-8',
                ],
                'body'    => $body,
            ]);

            if ($request->getStatusCode() !== 200) {
                throw new CartayException('Invalid response from the cart server.');
            }

            // TODO: Determine the cart is updated, added or failed.
            // $data = (array) json_decode($request->getBody()->getContents(), true);

            return true;
        } catch (GuzzleException $exception) {
            throw new CartayException($exception->getMessage());
        }
    }

    /**
     * @inheritDoc
     */
    public function update(string $itemId, array $params): bool
    {
        if (empty($params['quantity']) || !is_int($params['quantity'])) {
            throw new CartayException('Please provide valid quantity.');
        }

        $cart      = $this->cart();
        $csrfToken = $this->getCartCSRFToken();

        $items = array_filter($cart->getItems(), fn($item) => (string) $item['id'] === $itemId);
        $item = count($items) ? $items[0] : null;
        if (!$item) {
            throw new CartayException('Unable to find the item on the cart.');
        }

        $url  = 'https://cart.1688.com/ajax/calculate.htm';
        $data = json_encode([
            [
                "cart"     => [
                    [
                        "cartId"   => $item['id'],
                        "quantity" => $params['quantity'],
                        "flow"     => "general",
                    ],
                ],
                "sellerId" => $item['sellerId'] ?? 1,
                "operate"  => [$item['id']],
                "type"     => "update",
            ],
        ]);

        $body = http_build_query([
            'data'         => $data,
            'purchaseType' => 'main_purchase_type',
            '_csrf_token'  => $csrfToken,
        ]);

        try {
            $request = $this->client()->post($url, [
                'headers' => [
                    'Content-Type' => 'application/x-www-form-urlencoded; charset=UTF-8',
                ],
                'body'    => $body,
            ]);

            if ($request->getStatusCode() !== 200) {
                throw new CartayException('Invalid response from the cart server.');
            }
            $data = (array) json_decode($request->getBody()->getContents(), true);

            return isset($data['success']) && $data['success'] === 'true';
        } catch (GuzzleException $exception) {
            throw new CartayException($exception->getMessage());
        }
    }

    /**
     * @inheritDoc
     */
    public function remove(string $itemId): bool
    {
        $cart      = $this->cart();
        $csrfToken = $this->getCartCSRFToken();

        $items = array_values(array_filter($cart->getItems(), fn($item) => (string) $item['id'] === $itemId));

        $item = $items[0] ?? null;

        if (!is_array($item)) {
            throw new CartayException('Unable to find the item on the cart.');
        }

        $url  = 'https://cart.1688.com/ajax/calculate.htm';
        $data = json_encode([
            [
                "cart"     => [
                    [
                        "cartId"   => $item['id'],
                        "quantity" => $item['sku']['quantity'] ?? 1,
                        "flow"     => "general",
                    ],
                ],
                "sellerId" => $item['sellerId'] ?? 1,
                "operate"  => [$item['id']],
                "type"     => "delete",
            ],
        ]);

        $body = http_build_query([
            'data'         => $data,
            'purchaseType' => 'main_purchase_type',
            '_csrf_token'  => $csrfToken,
        ]);

        try {
            $request = $this->client()->post($url, [
                'headers' => [
                    'Content-Type' => 'application/x-www-form-urlencoded; charset=UTF-8',
                ],
                'body'    => $body,
            ]);

            if ($request->getStatusCode() !== 200) {
                throw new CartayException('Invalid response from the cart server.');
            }
            $data = (array) json_decode($request->getBody()->getContents(), true);

            return isset($data['success']) && $data['success'] === 'true';
        } catch (GuzzleException $exception) {
            throw new CartayException($exception->getMessage());
        }
    }

    /**
     * @inheritDoc
     */
    public function orders(int $page = 1, array $filters = []): Orders
    {
        try {
            $url     = 'https://trade.1688.com/order/buyer_order_list.htm';
            $request = $this->client()->get($url, [
                'query' => [
                    'page' => $page,
                ],
            ]);

            if ($request->getStatusCode() !== 200) {
                $this->orders->setStatus('error', 'Invalid server response. Response Code: ' . $request->getStatusCode());

                return $this->orders;
            }

            $crawler   = new Crawler($request->getBody()->getContents());
            $ordersEls = $crawler->filter('#listBox > ul > li');

            if (!$ordersEls->count()) {
                $this->orders->setStatus('error', 'Unable to parse orders.');

                return $this->orders;
            }

            $buyerId   = null;
            $buyerIdEl = $crawler->filter('#refundCheckUrl')->first();
            if ($buyerIdEl->count()) {
                $buyerIdUrl = $buyerIdEl->attr('value');
                $parseUrl   = parse_url($buyerIdUrl, PHP_URL_QUERY);
                parse_str($parseUrl, $parseQuery);

                $buyerId = $parseQuery['buyer_id'] ?? null;
            }

            $buyerMemberId   = null;
            $buyerMemberIdEl = $crawler->filter('#ts-memberid')->first();
            if ($buyerMemberIdEl->count()) {
                $buyerMemberId = $buyerMemberIdEl->attr('value');
            }

            $ordersEls->each(function (Crawler $crawler) use ($buyerId, $buyerMemberId) {
                $orderIdEl = $crawler->filter('input.tradeId')->first();
                $orderId   = $orderIdEl->count() ? trim($orderIdEl->attr('value')) : null;

                $sellerIdEl = $crawler->filter('[data-tplogo-memberid]')->first();
                $sellerId   = $sellerIdEl->count() ? $sellerIdEl->attr('data-tplogo-memberid') : null;

                $createdAt   = null;
                $createdAtEl = $crawler->filter('.order-title > .top > .left .date')->first();
                if ($createdAtEl->count() && !empty($createdAtEl->text())) {
                    $createdAt = trim($createdAtEl->text());
                }

                $buyerAccount   = null;
                $buyerAccountEl = $crawler->filter('.order-detail > .subtitle > .buyer-name > .text-aux')->first();
                if ($buyerAccountEl->count() && !empty($buyerAccountEl->text())) {
                    $buyerAccount = trim($buyerAccountEl->text());
                }

                $total   = null;
                $totalEl = $crawler->filter('.orderInfo td.s6 > .total')->first();
                if ($totalEl->count() && !empty($totalEl->text())) {
                    $total = str_replace(',', '', trim($totalEl->text()));
                }

                $realTotal   = null;
                $realTotalEl = $crawler->filter('.orderInfo td.s6 > .sum-payment-original')->first();
                if ($realTotalEl->count() && !empty($realTotalEl->text())) {
                    $realTotal = str_replace(',', '', trim($realTotalEl->text()));
                }

                $shippingCharge  = null;
                $freightChargeEl = $crawler->filter('.orderInfo td.s6 > .fare')->first();
                if ($freightChargeEl->count() && !empty($freightChargeEl->text())) {
                    preg_match('/[0-9.]+/', $freightChargeEl->text(), $chargeMatches);
                    $shippingCharge = $chargeMatches[0] ?? null;
                }

                $sellerUrl    = null;
                $sellerName   = null;
                $sellerNameEl = $crawler->filter('.order-title > .top > .left .bannerCorp');
                if ($sellerNameEl->count()) {
                    $sellerName = trim($sellerNameEl->attr('data-copytitle'));
                    $sellerUrl  = trim($sellerNameEl->attr('href'));
                }

                $orderStatus   = null;
                $orderStatusEl = $crawler->filter('.orderInfo td.s7 > [class|="status"]')->first();
                if ($orderStatusEl->count() && !empty($orderStatusEl->text())) {
                    $orderStatus = trim($orderStatusEl->text());
                }

                $refundStatus = null; // TODO: Collect.

                $attributes = [
                    'id'              => $orderId,
                    'total'           => $total,
                    'real_total'      => $realTotal,
                    'shipping_charge' => $shippingCharge,
                    'buyer_id'        => $buyerId,
                    'buyer_member_id' => $buyerMemberId,
                    'buyer_account'   => $buyerAccount,
                    'created_at'      => $createdAt,
                    'status'          => $orderStatus,
                    'refund_status'   => $refundStatus,
                    'seller'          => [
                        'id'   => $sellerId,
                        'name' => $sellerName,
                        'url'  => $sellerUrl,
                    ],
                    'skus'            => $this->parseOrderSkus($crawler),
                ];

                $this->orders->add(new OrderItem($attributes));
            });

            $currentPageEl = $crawler->filter('.paging-jumper > div > .paging-point > .current')->first();
            $totalPagesEl  = $crawler->filter('.paging-jumper > div > .paging-point > .pages-total')->first();

            $this->orders->currentPage = $currentPageEl->count() ? (int) $currentPageEl->text() : $this->orders->currentPage;
            $this->orders->pageCount   = $totalPagesEl->count() ? (int) $totalPagesEl->text() : $this->orders->pageCount;

            $this->orders->setStatus('success', 'Order list has been fetched.');
        } catch (GuzzleException $exception) {
            $this->orders->setStatus('error', 'Guzzle Exception: ' . $exception->getMessage());
        }

        return $this->orders;
    }

    /**
     * @inheritDoc
     */
    public function tracking(string $orderId, array $params = []): Tracking
    {
        $trackingModel = new Tracking;
        try {
            $url     = "https://trade.1688.com/order/new_step_order_detail.htm";
            $request = $this->client()->get($url, [
                'query' => [
                    'orderId' => $orderId,
                ],
            ]);
            if ($request->getStatusCode() !== 200) {
                $trackingModel->setStatus(
                    'error',
                    'Invalid response from server. Response Code: ' . $request->getStatusCode()
                );

                return $trackingModel;
            }
            $crawler = new Crawler($request->getBody()->getContents());
            $orderEl = $crawler->filter('[data-logisticsorder]')->first();

            if (!$orderEl->count()) {
                $trackingModel->setStatus('error', 'Unable to parse the tracking code.');

                return $trackingModel;
            }

            $orderInfo = $orderEl->attr('data-logisticsorder');
            $orderInfo = json_decode($orderInfo, true);
            $tracking  = $orderInfo['logisticsBillNo'] ?? null;

            if (!$tracking) {
                $trackingModel->setStatus('error', 'Unable to parse the tracking code.');

                $trackingModel->setAttributes((array) $orderInfo);

                return $trackingModel;
            }

            $trackingModel->setCode($tracking);
            $trackingModel->setStatus('success', 'Tracking code has been parsed.');
            $trackingModel->setAttributes((array) $orderInfo);

            return $trackingModel;
        } catch (GuzzleException $exception) {
            $trackingModel->setStatus('error', 'Guzzle Exception: ' . $exception->getMessage());

            return $trackingModel;
        }
    }

    /**
     * Parse cart item and set.
     *
     * @return void
     */
    private function parseCartItems(): void
    {
        try {
            $url     = 'https://cart.1688.com/ajax/purchaseRender.jsx';
            $request = $this->client()->post($url, [
                'headers'     => [
                    'Accept'     => 'application/json; charset=utf8',
                    'Referer'    => 'https://cart.1688.com/cart.htm',
                    'User-Agent' => 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/106.0.0.0 Safari/537.36',
                ],
                'query'       => [
                    '_input_charset' => 'utf-8',
                ],
                'form_params' => [
                    '_csrf_token'  => $this->getCartCSRFToken(),
                    'purchaseType' => 'main_purchase_type',
                    'platformType' => 'PC',
                ],
            ]);

            if ($request->getStatusCode() !== 200) {
                $this->cart->setStatus(
                    'error',
                    'Invalid response from server. Response Code: ' . $request->getStatusCode()
                );

                return;
            }

            $response = $request->getBody()->getContents();
            $response = json_decode($response, true);
            $response = json_decode($response['data'], true);
            $results  = Collection::make($response['data'])->groupBy('type');
            $root     = Collection::make($response['data']['root_000000']['fields']['validSkuCargos']);
            $checkoutPayload = [];

            $items = $results['item']->map(function ($item) use ($results, $root, &$checkoutPayload) {
                $detail = $results['item_group']->where('id', '=', $item['fields']['offerId'])->first();
                $shop   = $results['shop_top']->where('id', '=', $item['fields']['sellerId'])->first();
                $attrs  = $root->where('cartId', '=', $item['id'])->first();
                    $checkoutPayload[] = [
                        "cartId" => $item['fields']['cartId'],
                        "flow" => "general",
                        "offerId" => $item['fields']['offerId'],
                        "quantity" => $item['fields']['quantity'],
                        "skuId" => $item['fields']['skuId'] ?? null,
                        "specId" => $item['fields']['specId'] ?? null,
                    ];

                return [
                    'id'                 => $item['id'],
                    'seller'             => [
                        'id'          => $item['fields']['sellerId'],
                        'memberId'    => $shop['fields']['memberId'],
                        'companyName' => $shop['fields']['companyName'],
                        'companyUrl'  => $shop['fields']['link'],
                    ],
                    'sellerId'           => $item['fields']['sellerId'],
                    'productId'          => $item['fields']['offerId'],
                    'quantityMin'        => $item['fields']['minQuantity'],
                    'quantityMax'        => $item['fields']['maxQuantity'],
                    'title'              => $detail['fields']['title'],
                    'link'               => $detail['fields']['link'],
                    'image'              => $detail['fields']['pic'],
                    'thumb'              => str_replace('.jpg', '.80x80.jpg', $detail['fields']['pic']),
                    'sku'                => [
                        'id'          => $item['id'] ?? null,
                        'quantity'    => $item['fields']['quantity'],
                        'quantityMin' => $item['fields']['minQuantity'],
                        'quantityMax' => $item['fields']['maxQuantity'],
                        'skuId'       => $item['fields']['skuId'] ?? null,
                        'image'       => $detail['fields']['pic'],
                        'attrs'       => array_map(function ($attr) {
                            return [
                                'key'   => $attr['specName'],
                                'value' => $attr['specValue'],
                            ];
                        }, $attrs['skuSpecs']),
                        'skuName'     => $item['fields']['skuTitle'] ?? null,
                        'unit'        => $item['fields']['unit'], // TODO: fix me, translate to english.
                        'price'       => floatval($item['fields']['finalUnitPrice']),
                    ],
                    'inc_freight_cost' => false,
                    'current_freight'    => [],
                    'available_freights' => [],
                ];
            });

            $this->postCheckout($checkoutPayload);
            $freights = $this->parseFreights();
            foreach ($items as $item) {
                $freight = $freights->where('key', $item['productId'])->first();
                if ($freight) {
                    $item['inc_freight_cost'] = false;
                    $item['current_freight'] = [
                        'id'        => $freight['value']['serviceName'] ?? null,
                        'name'      => $freight['value']['displayName'] ?? null,
                        'cost'      => $freight['value']['fee'] ?? null,
                    ];
                }

                $this->cart->add(new CartItem($item));
            }

            $this->cart->limit = 200;

            $this->cart->setStatus(
                'success',
                'Cart items are fetched.'
            );

        } catch (GuzzleException $exception) {
            $this->cart->setStatus(
                'error',
                $exception->getMessage()
            );
        }
    }

    /**
     * Get CSRF Token.
     *
     * @return string|null
     * @throws \MoveOn\Cartay\Exceptions\CartayException
     */
    private function getCartCSRFToken(): string
    {
        if (!$this->crawler) {
            try {
                $url     = 'https://cart.1688.com/cart.htm';
                $request = $this->client()->get($url);

                if ($request->getStatusCode() !== 200) {
                    throw new CartayException('Unable to get CSRF Token.');
                } else {
                    $this->crawler = new Crawler($request->getBody()->getContents());
                }
            } catch (GuzzleException $exception) {
                throw new CartayException('Unable to get CSRF Token.');
            }
        }

        $csrfEl = $this->crawler->filter('#doc input[name="_csrf_token"]')->first();

        $token = $csrfEl->count() ? $csrfEl->attr('value') : null;

        if (empty($token)) {
            throw new CartayException('Unable to get CSRF Token.');
        }

        return $token;
    }

    /**
     * Parse SKUs.
     *
     * @param \Symfony\Component\DomCrawler\Crawler $crawler
     *
     * @return array
     */
    private function parseOrderSkus(Crawler $crawler): array
    {
        $skus = [];
        $crawler->filter('td.detail > table > tr')->each(function (Crawler $crawler) use (&$skus) {
            $productId     = null;
            $productLink   = null;
            $productImage  = null;
            $productLinkEl = $crawler->filter('td.s1 > a');
            if ($productLinkEl->count()) {
                $productLink    = $productLinkEl->attr('href');
                $productImageEl = $productLinkEl->filter('img')->first();

                if ($productImageEl->count()) {
                    $productImage = $productImageEl->attr('src');
                }

                preg_match('/(?<=\/offer\/)(.*)(?=.html)/', $productLink, $productIdMatches);

                $productId = $productIdMatches[0] ?? null;
            }

            $productTitle  = null;
            $productNameEl = $crawler->filter('td.s2 > div > a.productName');
            if ($productNameEl->count()) {
                $productTitle = $productNameEl->text();
            }

            $orderId     = null;
            $productIdEl = $crawler->filter('td.s2 > div > a.link-lesser');
            if ($productIdEl->count()) {
                $parseUrl = parse_url(trim($productIdEl->attr('href')), PHP_URL_QUERY);
                parse_str($parseUrl, $parseQuery);
                $orderId = $parseQuery['order_entry_id'] ?? null;
            }

            $price   = null;
            $priceEl = $crawler->filter('td.s3 > div[title]')->first();
            if ($priceEl->count()) {
                $price = trim($priceEl->attr('title'));
            }

            $price_real  = null;
            $priceRealEl = $crawler->filter('td.s3 > .b-price-amount > span')->first();
            if ($priceRealEl->count()) {
                $price_real = trim($priceRealEl->text());
            }

            $quantity   = 1;
            $quantityEl = $crawler->filter('td.s4 > div[title]')->first();
            if ($quantityEl->count()) {
                $quantity = (int) trim($quantityEl->attr('title'));
            }

            $attrs = [];
            $crawler->filter('td.s2 > .c > .trade-spec > .spec-item')->each(function (Crawler $crawler) use (&$attrs) {
                $content  = str_replace('; ', '', $crawler->text());
                $keyValue = explode(':', trim($content));

                $attrs[] = [
                    'key'   => $keyValue[0] ?? null,
                    'value' => $keyValue[1] ?? null,
                ];
            });

            $skus[] = [
                'id'         => $orderId,
                'product_id' => $productId,
                'title'      => $productTitle,
                'link'       => $productLink,
                'image'      => $productImage,
                'price'      => $price,
                'price_real' => $price_real,
                'quantity'   => $quantity,
                'sku_id'     => null,
                'attributes' => $attrs,
                'status'     => null,
            ];
        });

        return $skus;
    }

    /**
     * @param $checkoutPayload
     *
     * @throws \MoveOn\Cartay\Exceptions\CartayException
     */
    private function postCheckout($checkoutPayload):void {
        try {
            $orderRequest = $this->client()->post('https://order.1688.com/order/smart_make_order.htm', [
                'headers'     => [
                    'Referer'    => 'https://cart.1688.com/cart.htm',
                ],
                'query'       => [
                    'p' => 'cart',
                ],
                'form_params' => [
                    '_csrf_token'  => $this->getCartCSRFToken(),
                    'id' => json_encode($checkoutPayload),
                    'p' => 'cart',
                    "flow" => 'xxx',
                ],
            ]);
            if ($orderRequest->getStatusCode() !== 200) {
                throw new CartayException('Unable to make checkout post request.');
            } else {
                $orderDom = $orderRequest->getBody()->getContents();
                $this->crawler = new Crawler($orderDom);
            }
        } catch (GuzzleException $exception) {
            $this->cart->setStatus(
                'error',
                $exception->getMessage()
            );
        }
    }

    /**
     * @return \Illuminate\Support\Collection
     */
    private function parseFreights(): Collection {
        $fees = [];
        $this->crawler->filter('.freight')->each(function (Crawler $node) use (&$fees) {
            $trNode = $node->closest('tr');
            $trDataSource = ($trNode->attr('data-source'));
            $orderItem = json_decode(str_replace("'", '"', $trDataSource), true);
            $freight = json_decode($node->attr('data-freight'), true);
            if ($freight) {
                $fees[] = [
                    "key"   => $orderItem['offerId'],
                    "value" => [
                        'serviceName' => $freight['freightId'],
                        'displayName' => '',
                        "fee"       => trim($freight['fee']),
                    ],
                ];
            }
        });

        return collect($fees);
    }
}
