<?php

namespace MoveOn\Cartay;

use MoveOn\Cartay\Contracts\Account;
use MoveOn\Cartay\Contracts\Driver;
use MoveOn\Cartay\Exceptions\CartayException;
use MoveOn\Cartay\Helpers\CookieManager;

class Cartay implements Driver
{
    /**
     * @var \MoveOn\Cartay\Contracts\Account
     */
    protected Account $account;

    /**
     * @var \MoveOn\Cartay\Contracts\Driver
     */
    protected Driver $driver;

    /**
     * @var array
     */
    public static array $customDrivers = [];

    /**
     * Master constructor.
     *
     * @param \MoveOn\Cartay\Contracts\Account $account
     *
     * @throws \Exception
     */
    public function __construct(Account $account)
    {
        $this->account = $account;
        $this->setDriver();
    }

    /**
     * All core drivers.
     *
     * @return string[]
     */
    public static function coreDrivers(): array
    {
        return [
            'alibaba1688' => '\MoveOn\Cartay\Drivers\Alibaba1688',
            'aliexpress'  => '\MoveOn\Cartay\Drivers\AliExpress',
            'taobao'      => '\MoveOn\Cartay\Drivers\Taobao',
        ];
    }

    /**
     * Register custom drivers.
     *
     * @param array $drivers
     */
    public static function registerCustomDrivers(array $drivers)
    {
        static::$customDrivers = array_merge(static::$customDrivers, $drivers);
    }

    /**
     * Get all available driver.
     *
     * @return array
     */
    public static function getAllDrivers(): array
    {
        return array_merge(self::coreDrivers(), static::$customDrivers);
    }

    /**
     * Set the driver.
     *
     * @param string|null $name
     *
     * @throws \MoveOn\Cartay\Exceptions\CartayException
     */
    public function setDriver(?string $name = null)
    {
        $drivers = $this->getAllDrivers();
        $name    = $name ?? $this->account->getDriver();
        $driver  = $drivers[$name] ?? null;

        if (empty($driver)) {
            throw new CartayException("The driver: \"$name\" is not registered.");
        }

        $this->driver = new $driver($this->account);
    }

    /**
     * The cart.
     *
     * @return \MoveOn\Cartay\Cart
     */
    public function cart(): Cart
    {
        return $this->driver->cart();
    }

    /**
     * Get orders.
     *
     * @param int $page
     * @param array $filters
     *
     * @return \MoveOn\Cartay\Orders
     */
    public function orders(int $page = 1, array $filters = []): Orders
    {
        return $this->driver->orders($page, $filters);
    }

    /**
     * Get tracking by Order ID.
     *
     * @param string $orderId
     * @param array $params
     *
     * @return \MoveOn\Cartay\Tracking
     */
    public function tracking(string $orderId, array $params = []): Tracking
    {
        return $this->driver->tracking($orderId, $params);
    }

    /**
     * Add item to the cart.
     *
     * $skus = [
     *     ['id' => '1234', 'qty' => 3]
     * ]
     *
     * @param string $link
     * @param array $skus
     *
     * @return bool
     * @throws \MoveOn\Cartay\Exceptions\CartayException
     */
    public function add(string $link, array $skus): bool
    {
        return $this->driver->add($link, $skus);
    }

    /**
     * Update cart item.
     *
     * Example for $params: ['quantity' => 1]
     *
     * @param string $itemId
     * @param array $params
     *
     * @return bool
     * @throws \MoveOn\Cartay\Exceptions\CartayException
     */
    public function update(string $itemId, array $params): bool
    {
        return $this->driver->update($itemId, $params);
    }

    /**
     * Remove item from the cart.
     *
     * @param string $itemId
     *
     * @return bool
     * @throws \MoveOn\Cartay\Exceptions\CartayException
     */
    public function remove(string $itemId): bool
    {
        return $this->driver->remove($itemId);
    }

    /**
     * Clear cookies.
     *
     * @return bool
     */
    public function clearCookies(): bool
    {
        return CookieManager::clearCookies($this->account);
    }
}
