<?php

namespace MoveOn\Cartay\Drivers;

use DateTime;
use GuzzleHttp\Client;
use GuzzleHttp\Cookie\CookieJar;
use GuzzleHttp\Cookie\FileCookieJar;
use GuzzleHttp\Cookie\SetCookie;
use GuzzleHttp\Exception\GuzzleException;
use MoveOn\Cartay\Cart;
use MoveOn\Cartay\CartItem;
use MoveOn\Cartay\Exceptions\CartayException;
use MoveOn\Cartay\OrderItem;
use MoveOn\Cartay\Orders;
use MoveOn\Cartay\Tracking;

class AliExpress extends Driver
{
    /**
     * Cookies Jar.
     *
     * @var \GuzzleHttp\Cookie\CookieJar|null
     */
    private ?CookieJar $jar = null;

    /**
     * The client.
     *
     * @return \GuzzleHttp\Client
     */
    private function client(): Client
    {
        $this->setCookieJar();

        if (empty($this->headers['Referer']) || empty($this->headers['referer'])) {
            $this->headers['Referer'] = 'https://www.aliexpress.com/';
        }

        if (empty($this->headers['User-Agent']) || empty($this->headers['user-agent'])) {
            $this->headers['User-Agent'] = 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/92.0.4515.131 Safari/537.36';;
        }

        return new Client([
            'cookies'     => $this->jar,
            'headers'     => $this->headers,
            'http_errors' => false,
            'timeout'     => 30,
        ]);
    }

    /**
     * Parse product ID from the link.
     *
     * @param string $link
     *
     * @return mixed|null
     */
    public static function parseIdFromLink(string $link): ?string
    {
        preg_match('/(?<=\/item\/)(.*)(?=.html)/', $link, $matches);
        if (!is_array($matches) || count($matches) <= 0) {
            return null;
        }

        $withSlug = explode('/', $matches[0]);

        return $withSlug[1] ?? $withSlug[0];
    }

    /**
     * @inheritDoc
     */
    public function cart(): Cart
    {
        try {
            $cartApiUrl     = 'https://shoppingcart.aliexpress.com/api/1.0/cart.do';
            $cartApiRequest = $this->client()->post($cartApiUrl, [
                'json' => [
                    'selectAll'    => true,
                    'selected'     => '',
                    '_csrf_token_' => $this->getToken(),
                ],
            ]);

            if ($cartApiRequest->getStatusCode() !== 200) {
                $this->cart->setStatus(
                    'error',
                    'Invalid response from the cart server. HTTP Status Code: ' . $cartApiRequest->getStatusCode()
                );

                return $this->cart;
            }

            $response = json_decode($cartApiRequest->getBody()->getContents(), true);
            if (!is_array($response) || !isset($response['stores'])) {
                $this->cart->setStatus(
                    'error',
                    'Unable to get cart items from the API.'
                );

                return $this->cart;
            }

            foreach ($response['stores'] as $store) {
                foreach ($store['storeList'] as $store) {
                    $cartItem = new CartItem();
                    $seller   = [
                        'id'          => $store['sellerId'] ?? null,
                        'memberId'    => null,
                        'companyName' => $store['name'] ?? null,
                        'companyUrl'  => $store['storeUrl'] ?? null,
                    ];
                    foreach ($store['products'] as $product) {
                        $cartItem->id          = $product['itemId'] ?? null;
                        $cartItem->sellerId    = $seller['id'];
                        $cartItem->seller      = $seller;
                        $cartItem->productId   = $product['productId'] ?? null;
                        $cartItem->quantityMin = $product['minimum'] ?? null;
                        $cartItem->quantityMax = $product['maximum'] ?? null;
                        $cartItem->title       = $product['title'] ?? null;
                        $cartItem->link        = $product['detailUrl'] ?? null;
                        $imageUrl              = $product['mainImg'] ?? null;
                        $cartItem->image       = $imageUrl;
                        $cartItem->thumb       = $imageUrl ? str_replace('.jpg', '.jpg_50x50.jpg', $imageUrl) : null;

                        $attrs = array_map(function ($attr) {
                            return [
                                'keyId'   => $attr['propertyNameId'],
                                'key'     => $attr['propertyName'],
                                'valueId' => $attr['propertyValues'][0]['propertyValueId'],
                                'value'   => $attr['propertyValues'][0]['propertyValue'],
                            ];
                        }, (array) $product['sku']);

                        $cartItem->sku = [
                            'id'          => $skuProfile['cartId'] ?? null,
                            'quantity'    => $product['count'] ?? null,
                            'quantityMin' => $cartItem->quantityMin ?? null,
                            'quantityMax' => $cartItem->quantityMax ?? null,
                            'skuId'       => null,
                            'image'       => $imageUrl,
                            'attrs'       => $attrs,
                            'skuName'     => $product['skuAttr'] ?? null,
                            'unit'        => null, // piece, lot, box, kg
                            'price'       => $product['cost']['price']['amount'],
                        ];

                        $logistics       = $product['logistics'] ?? [];
                        $availableList   = $logistics['availableFreightServices'] ?? [];
                        $currentFreight  = [];
                        $availableMapped = array_map(function ($freight) use ($logistics, &$currentFreight) {
                            preg_match('/[0-9.]+/', ($freight['freightCost'] ?? ''), $chargeMatches);

                            $meta = $freight;
                            if (array_key_exists('serviceName', $meta)) {
                                unset($meta['serviceName']);
                            }
                            if (array_key_exists('displayName', $meta)) {
                                unset($meta['displayName']);
                            }
                            if (array_key_exists('isTracked', $meta)) {
                                unset($meta['isTracked']);
                            }

                            $map = [
                                'id'        => $freight['serviceName'] ?? null,
                                'name'      => $freight['displayName'] ?? null,
                                'cost'      => $chargeMatches[0] ?? '0.00',
                                'trackable' => $freight['isTracked'] ?? null,
                                'meta'      => $meta,
                            ];

                            if ($map['id'] === $logistics['serviceName']) {
                                $currentFreight = $map;
                            }

                            return $map;
                        }, (array) $availableList);

                        $cartItem->current_freight    = $currentFreight;
                        $cartItem->available_freights = $availableMapped;

                        $this->cart->add($cartItem);
                    }
                }
            }

            $this->cart->setStatus(
                'success',
                'Cart items are fetched.'
            );
        } catch (GuzzleException $exception) {
            $this->cart->setStatus('error', $exception->getMessage());
        }

        $this->cart->limit = 100;

        return $this->cart;
    }

    /**
     * @inheritDoc
     */
    public function add(string $link, array $skus): bool
    {
        $productId      = self::parseIdFromLink($link);
        $defaultCountry = $this->account->getOptions()['country'] ?? 'BD';

        if (empty($productId)) {
            throw new CartayException('Unable to parse product id from the link.');
        }

        try {
            $client         = $this->client(); // Keep this order, otherwise `prepareLink()` method unable to get `$jar` property.
            $productLink    = $this->prepareLink($link);
            $productRequest = $client->get($productLink);
            if ($productRequest->getStatusCode() !== 200) {
                throw new CartayException('Invalid response from the product server. Code:' . $productRequest->getStatusCode());
            }

            $content    = $productRequest->getBody()->getContents();
            $parsedSKUs = $this->getItemSKUs($content);
            if (count($parsedSKUs) === 0) {
                throw new CartayException('Unable to parse SKUs from the product page.');
            }

            $added = 0;
            foreach ($skus as $singleSku) {
                $skuKey  = array_search($singleSku['id'], array_column($parsedSKUs, 'skuId'));
                $skuData = $parsedSKUs[$skuKey] ?? null;

                if (!is_array($skuData)) {
                    throw new CartayException('Unable to find SKU from the SKU list.');
                }

                $cartApiUrl  = 'https://shoppingcart.aliexpress.com/addToShopcart4Js.htm';
                $cartRequest = $this->client()->get($cartApiUrl, [
                    'query' => [
                        'productId'         => $productId,
                        'quantity'          => $singleSku['qty'],
                        'country'           => $singleSku['country'] ?? $defaultCountry,
                        'company'           => $singleSku['shipping'] ?? null,
                        'promiseId'         => '',
                        'cartfrom'          => 'main_detail',
                        'skuAttr'           => $skuData['skuAttr'],
                        'skuId'             => $skuData['skuId'],
                        '_csrf_token_'      => $this->getToken(),
                        'carAdditionalInfo' => '{}',
                        'callback'          => '__zoro_request_1',
                    ],
                ]);

                if ($cartRequest->getStatusCode() !== 200) {
                    throw new CartayException('Invalid response from the cart server. Code:' . $cartRequest->getStatusCode());
                }

                $added++;
            }

            return $added > 0;
        } catch (GuzzleException $exception) {
            throw new CartayException($exception->getMessage());
        }
    }

    /**
     * @inheritDoc
     */
    public function update(string $itemId, array $params): bool
    {
        // Example input: ['quantity' => 1] or ['shipping' => 'SHIPPING_SERVICE_NAME'] or ['delete' => 'yes']

        try {
            $cartPageUrl = 'https://shoppingcart.aliexpress.com/shopcart/shopcartDetail.htm';
            $cartPage    = $this->client()->get($cartPageUrl);
            if ($cartPage->getStatusCode() !== 200) {
                throw new CartayException('Invalid response from the cart server. Code:' . $cartPage->getStatusCode());
            }

            $action = 'CHANGE_ITEM_QUANTITY';
            $field  = 'quantity';
            $value  = $params['quantity'] ?? null;
            if (isset($params['shipping'])) {
                $action = 'CHANGE_ITEM_LOGISTICS_SERVICE';
                $field  = 'serviceName';
                $value  = $params['shipping'];
            } elseif (isset($params['delete'])) {
                $action = 'DELETE_ITEMS';
                $value  = 0;
            }

            $cartApiUrl  = 'https://shoppingcart.aliexpress.com/api/1.0/cart.do';
            $cartRequest = $this->client()->post($cartApiUrl, [
                'json' => [
                    'action'       => $action,
                    'selected'     => '',
                    'updates'      => [
                        [
                            'itemId' => $itemId,
                            $field   => $value,
                        ],
                    ],
                    '_csrf_token_' => $this->getToken(),
                ],
            ]);

            if ($cartRequest->getStatusCode() !== 200) {
                throw new CartayException('Unable to update the cart item. Code:' . $cartRequest->getStatusCode());
            }

            return true;
        } catch (GuzzleException $exception) {
            throw new CartayException($exception->getMessage());
        }
    }

    /**
     * @inheritDoc
     */
    public function remove(string $itemId): bool
    {
        return $this->update($itemId, ['delete' => 'yes']);
    }

    /**
     * @inheritDoc
     */
    public function orders(int $page = 1, array $filters = []): Orders
    {
        try {
            $url     = 'https://m.aliexpress.com/api/user/orders';
            $request = $this->client()->post($url, [
                'headers' => [
                    'Referer'         => 'https://m.aliexpress.com/orderlist.html?spm=a2g0n.new_account_index',
                    'User-Agent'      => 'Mozilla/5.0 (Linux; Android 8.0; Pixel 2 Build/OPD3.170816.012) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/95.0.4638.69 Mobile Safari/537.36',
                    'amp-same-origin' => 'true',
                ],
                'query'   => [
                    'currentPage'             => $page ?? 1,
                    'pageSize'                => 20,
                    'timeZoneOffsetOfMinutes' => '360',
                    '__amp_source_origin'     => 'https://m.aliexpress.com',
                ],
            ]);

            if ($request->getStatusCode() !== 200) {
                $this->orders->setStatus('error', 'Invalid response from the orders server. Code: ' . $request->getStatusCode());

                return $this->orders;
            }

            $response = $request->getBody()->getContents();
            $response = json_decode($response, true);
            $data     = $response['data'] ?? [];
            $orders   = $data['orderList'] ?? [];
            if (!is_array($orders) || count($orders) === 0) {
                $this->orders->setStatus('error', 'Invalid response from the orders server. Code: ' . $request->getStatusCode());

                return $this->orders;
            }

            foreach ($orders as $order) {
                $orderTime = DateTime::createFromFormat('U', ceil($order['gmtTradeCreate'] / 1000));
                $orderTime = $orderTime->format('Y-m-d H:i:s');

                $skus = array_map(function ($sku) use ($order) {
                    $attrs = array_map(function ($attr) {
                        return [
                            'key'      => $attr['pName'],
                            'value'    => $attr['pValue'],
                            'key_id'   => $attr['pId'],
                            'value_id' => $attr['pValueId'],
                        ];
                    }, $sku['productSku']);

                    return [
                        'id'         => $sku['subOrderId'],
                        'product_id' => $sku['productId'],
                        'title'      => $sku['productName'],
                        'link'       => 'https://www.aliexpress.com/item/' . $sku['productId'] . '.html',
                        'image'      => str_replace('.jpg_220x220.jpg', '.jpg', $sku['smallPhotoFullPath']),
                        'price'      => $sku['productBuyerLocalPrice']['value'],
                        'price_real' => null,
                        'quantity'   => $sku['productCount'],
                        'sku_id'     => count($attrs) ? implode(',', array_column($attrs, 'value_id')) : null,
                        'attributes' => $attrs,
                        'status'     => $sku['subOrderStatus'],
                    ];
                }, $order['subList']);

                $attributes = [
                    'id'              => $order['orderId'],
                    'total'           => $order['originOrderAmount']['value'],
                    'real_total'      => null,
                    'shipping_charge' => null,
                    'buyer_id'        => null,
                    'buyer_member_id' => null,
                    'buyer_account'   => $order['buyerAliId'],
                    'created_at'      => $orderTime,
                    'status'          => $order['orderStatus'],
                    'refund_status'   => null,
                    'seller'          => [
                        'id'   => $order['sellerAliId'],
                        'name' => null,
                        'url'  => "https://m.aliexpress.com/store/storeHome.htm?sellerAdminSeq=" . $order['sellerAliId'],
                    ],
                    'skus'            => $skus,
                ];

                $this->orders->add(new OrderItem($attributes));
            }

            $total = $data['totalNum'] ?? 0;

            $this->orders->currentPage = $page;
            $this->orders->pageCount   = $total > 0 ? ceil($total / 20) : 0;
            $this->orders->setStatus('success', 'Order list has been fetched.');
        } catch (GuzzleException $exception) {
            $this->orders->setStatus('error', $exception->getMessage());
        }

        return $this->orders;
    }

    /**
     * @inheritDoc
     */
    public function tracking(string $orderId, array $params = []): Tracking
    {
        $trackingModel = new Tracking;
        try {
            $request = $this->client()->get('https://ilogisticsaddress.aliexpress.com/ajax_logistics_track.htm', [
                'query' => [
                    'orderId'  => $orderId,
                    'callback' => '_',
                ],
            ]);

            if ($request->getStatusCode() !== 200) {
                return $trackingModel;
            }

            $trackingData = $request->getBody()->getContents();
            $trackingData = substr($trackingData, 3, -1);
            $trackingData = json_decode($trackingData, true);

            if (!is_array($trackingData) || !isset($trackingData['tracking'])) {
                $trackingModel->setStatus('error', 'Unable to parse the tracking code.');

                return $trackingModel;
            }

            $trackingCode = $trackingData['tracking'][0] ?? [];
            $trackingCode = $trackingCode['mailNo'] ?? null;

            if (empty($trackingCode)) {
                $trackingModel->setStatus('error', 'Unable to parse the tracking code.');

                return $trackingModel;
            }

            $trackingModel->setCode($trackingCode);
            $trackingModel->setStatus('success', 'Tracking code has been parsed.');
            $trackingModel->setAttributes((array) $trackingData['tracking']);

            return $trackingModel;
        } catch (GuzzleException $exception) {
            $trackingModel->setStatus('error', 'Guzzle Exception: ' . $exception->getMessage());

            return $trackingModel;
        }
    }

    /**
     * Get CSRF Token from page the page.
     *
     * @return string|null
     */
    private function getToken(): ?string
    {
        if (!$this->jar) {
            $this->setCookieJar();
        }

        $cookie = $this->jar->getCookieByName('acs_usuc_t');
        if (!$cookie) {
            return null;
        }
        $cookie = $cookie->toArray();

        $tokens = [];
        parse_str($cookie['Value'], $tokens);

        return $tokens['x_csrf'] ?? null;
    }

    /**
     * Get item SKUs.
     *
     * @param string $content
     *
     * @return array
     */
    private function getItemSKUs(string $content): array
    {
        $content   = preg_replace('~[\r\n\t]+~', '', $content);
        $startStr  = 'data: ';
        $endStr    = '"site":"glo"';
        $start     = strpos($content, $startStr);
        $start     = $start ? ($start + strlen($startStr)) : null;
        $end       = strpos($content, $endStr);
        $end       = $end ? ($end - ($start - strlen($endStr))) : null;
        $nodeValue = substr($content, $start, $end) . '}}';
        $modules   = json_decode($nodeValue, true);

        return is_array($modules) && isset($modules['skuModule']) ? $modules['skuModule']['skuPriceList'] : [];
    }

    /**
     * Set cookies jar.
     *
     * @return void
     */
    private function setCookieJar(): void
    {
        $this->jar = new FileCookieJar($this->cookiesFile, true);

        if ($this->jar->count() === 0 && count($this->cookies) > 0) {
            foreach ($this->cookies as $name => $value) {
                $this->jar->setCookie(new SetCookie([
                    'Name'   => $name,
                    'Value'  => $value,
                    'Domain' => '.aliexpress.com',
                ]));
            }
        }
    }

    /**
     * Prepare the link before request.
     *
     * @param string $link
     * @return string
     */
    private function prepareLink(string $link): string
    {
        $epnHash   = $this->account->getOptions()['epn_hash'] ?? null;
        $epnDomain = $this->account->getOptions()['epn_domain'] ?? 'allshops.me';

        if (!$epnHash) {
            return $link;
        }

        $query = http_build_query(['to' => $link]);

        // we need to clear cookies to fake epn that we are new user ;)
        $this->jar->clear($epnDomain);

        return "https://$epnDomain/redirect/cpa/o/$epnHash/?$query";
    }
}
