<?php

namespace MoveOn\Cartay\Drivers;

use MoveOn\Cartay\Cart;
use MoveOn\Cartay\Contracts\Account;
use MoveOn\Cartay\Contracts\Driver as DriverInterface;
use MoveOn\Cartay\Helpers\CookieManager;
use MoveOn\Cartay\Orders;

abstract class Driver implements DriverInterface
{
    /**
     * @var \MoveOn\Cartay\Contracts\Account
     */
    protected Account $account;

    /**
     * @var array
     */
    protected array $cookies = [];

    /**
     * @var array
     */
    protected array $headers = [];

    /**
     * @var \MoveOn\Cartay\Cart
     */
    protected Cart $cart;

    /**
     * @var \MoveOn\Cartay\Orders
     */
    protected Orders $orders;

    /**
     * @var string
     */
    protected string $cookiesFile;

    /**
     * Driver constructor.
     *
     * @param \MoveOn\Cartay\Contracts\Account $account
     */
    public function __construct(Account $account)
    {
        $this->account     = $account;
        $this->cart        = new Cart;
        $this->orders      = new Orders;
        $this->cookiesFile = CookieManager::generateCookiePath($account);

        $this->setHeaders($account->getHeaders());
        $this->setCookies($account->getCookies());
    }

    /**
     * Set cookies.
     *
     * Accept key-value pairs only.
     *
     * Example: ['CookieName' => 'CookieValue']
     *
     * @param array $cookies
     */
    public function setCookies(array $cookies): void
    {
        $this->cookies = array_replace($this->cookies, $cookies);
    }

    /**
     * Set new headers.
     *
     * Accept key-value pairs only.
     *
     * Example: ['Header-Name' => 'Header-Value']
     *
     * @param array $headers
     */
    public function setHeaders(array $headers): void
    {
        $this->headers = array_replace($this->headers, $headers);
    }

    /**
     * Remove headers.
     *
     * @param array $headers
     */
    public function removeHeaders(array $headers): void
    {
        foreach ($headers as $header) {
            unset($this->headers[$header]);
        }
    }

    /**
     * Remove cookies.
     *
     * @param array $cookies
     */
    public function removeCookies(array $cookies): void
    {
        foreach ($cookies as $cookie) {
            unset($this->headers[$cookie]);
        }
    }

    /**
     * Set account.
     *
     * @param \MoveOn\Cartay\Contracts\Account $account
     * @param bool $clean
     */
    public function setAccount(Account $account, bool $clean = false): void
    {
        if ($clean) {
            $this->removeHeaders(array_keys($this->account->getHeaders()));
            $this->removeCookies(array_keys($this->account->getCookies()));
        }

        $this->account = $account;

        $this->setHeaders($account->getHeaders());
        $this->setCookies($account->getCookies());
    }
}
