<?php

namespace MoveOn\Cartay\Helpers;

use MoveOn\Cartay\Contracts\Account;

class CookieManager
{
    /**
     * Parsed cookies.
     *
     * @var array
     */
    protected array $cookies = [];

    /**
     * CookieManager constructor.
     *
     * @param string $cookie
     * @param string $domain
     */
    public function __construct(string $cookie, string $domain)
    {
        $this->cookies = self::parse($cookie, $domain);
    }

    /**
     * Parse cookies
     *
     * @param string $data
     *
     * @param ?string $domain
     *
     * @return array
     */
    public static function parse(string $data, ?string $domain = null): array
    {
        $cookies = explode('; ', $data);
        $items   = [];

        foreach ($cookies as $cookie) {
            $sep   = strpos($cookie, '=');
            $name  = substr($cookie, 0, $sep);
            $value = substr($cookie, ($sep + 1));

            $items[] = [
                "Name"     => trim($name),
                "Value"    => trim($value),
                "Domain"   => trim($domain),
                "Path"     => '/',
                "Max-Age"  => null,
                "Expires"  => null,
                "Secure"   => null,
                "Discard"  => null,
                "HttpOnly" => null,
            ];
        }

        return $items;
    }

    /**
     * Get cookies value.
     *
     * @param $name
     *
     * @return array|null
     */
    public function get($name): ?array
    {
        $index = array_search($name, array_column($this->cookies, 'Name'));

        return $this->cookies[$index] ?? null;
    }

    /**
     * Set cookie item.
     *
     * @param string $name
     * @param string $value
     * @param string $domain
     */
    public function set(string $name, string $value, string $domain): void
    {
        $cookie = [
            "Name"     => $name,
            "Value"    => $value,
            "Domain"   => $domain,
            "Path"     => '/',
            "Max-Age"  => null,
            "Expires"  => null,
            "Secure"   => null,
            "Discard"  => null,
            "HttpOnly" => null,
        ];

        $index = array_search($name, array_column($this->cookies, 'Name'));

        if ($index) {
            $this->cookies[$index] = $cookie;
        } else {
            $this->cookies[] = $cookie;
        }
    }

    /**
     * Generate cookie path.
     *
     * @param \MoveOn\Cartay\Contracts\Account $account
     *
     * @return string
     */
    public static function generateCookiePath(Account $account): string
    {
        $storagePath = rtrim(sys_get_temp_dir(), DIRECTORY_SEPARATOR);
        $fileName    = "cookie-{$account->getDriver()}-{$account->getAccountId()}";

        return $storagePath . '/' . md5($fileName);
    }

    /**
     * Clear cookies from storage.
     *
     * @param \MoveOn\Cartay\Contracts\Account $account
     *
     * @return bool
     */
    public static function clearCookies(Account $account): bool
    {
        $path = self::generateCookiePath($account);

        return file_exists($path) && unlink($path);
    }

    /**
     * Get the instance as an array.
     *
     * @return array
     */
    public function toArray()
    {
        return $this->cookies;
    }
}
