<?php

namespace MoveOn\Common\Modules;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use MoveOn\Common\Contracts\QueryFilterContract;
use MoveOn\Common\Traits\Makeable;

abstract class QueryFilter implements QueryFilterContract
{
    use Makeable;

    /**
     * The request object.
     *
     * @var Request
     */
    protected Request $request;

    /**
     * The builder instance.
     *
     * @var Builder
     */
    protected Builder $builder;

    /**
     * Create a new QueryFilters instance.
     *
     * @param Request $request
     */
    public function __construct(Request $request = null)
    {
        $this->request = $request ?? request();
    }

    /**
     * Apply the filters to the builder.
     *
     * @param Builder $builder
     * @return Builder
     */
    public function apply(Builder $builder): Builder
    {
        $this->builder = $builder;

        $filters = $this->filters();

        foreach ($filters as $name => $value) {
			$method = Str::camel($name);
            if (!method_exists($this, $method)) {
                continue;
            }

            if ($this->request->filled($name) && !empty($value)) {
                $this->$method($this->builder, $value);
            }
        }

        if (!array_key_exists("sort_order", $filters)) {
            $this->sortOrder($this->builder);
        }

        return $this->builder;
    }

    public function setRequest(Request $request): static
    {
        $this->request = $request;
        return $this;
    }

    /**
     * Get all request filters data.
     *
     * @return array
     */
    protected function filters(): array
    {
        $keys = $this->filterableKeys();
        $excludeKeys = $this->excludeKeys();

        if(!empty($keys)) $filterableData = $this->request->only($keys);
        else $filterableData = $this->request->all();

        if(!empty($excludeKeys)){
            foreach ($excludeKeys as $key){
                unset($filterableData[$key]);
            }
        }

        return $filterableData;
    }

    protected function sortOrder($query, $order = "desc")
    {
        $sortKey = in_array(
            $this->request->sort_by,
            $this->getAllowedSortKeys()
        ) ? $this->request->sort_by : 'created_at';

        return $query->orderBy($sortKey, $order);
    }

    private function getAllowedSortKeys(): array
    {
        return array_merge(['id', 'created_at', 'updated_at'], $this->allowedExtraSortKeys());
    }

    protected function id($query, string $value)
    {
        return $query->where("id", $value);
    }

    protected function created_at($query, $value)
    {
        if (array($value)) {
            return $query->whereBetween('created_at', $value);
        }

        return $query;
    }

    protected function allowedExtraSortKeys(): array
    {
        return [];
    }

    protected function filterableKeys(): array
    {
        return [];
    }

    protected function excludeKeys(): array
    {
        return [];
    }

    public static function filterableDetails(): array
    {
        return [
            "sort_by" => [
                "type"        => "select",
                "values"      => [
                    "id"         => "sort by id",
                    "created_at" => "sort by created_at",
                    "updated_at" => "sort by updated_at",
                ],
                "default" => "created_at",
                "description" => "Display in Order",
            ],

            "sort_order" => [
                "type"   => "select",
                "values" => [
                    "asc"  => "Ascending",
                    "desc" => "Descending",
                ],
            ],
        ];
    }
}