<?php

namespace MoveOn\EventBridgeSqsFifoQueue;

use Illuminate\Support\ServiceProvider;
use MoveOn\EventBridgeSqsFifoQueue\Queue\Connectors\SqsConsumerConnector;
use MoveOn\EventBridgeSqsFifoQueue\Queue\Deduplicators\Sqs;
use MoveOn\EventBridgeSqsFifoQueue\Queue\Deduplicators\Unique;
use MoveOn\EventBridgeSqsFifoQueue\Queue\Deduplicators\Content;
use MoveOn\EventBridgeSqsFifoQueue\Queue\Connectors\EventBridgeConnector;
use MoveOn\EventBridgeSqsFifoQueue\Queue\Connectors\SqsFifoConnector;
use Illuminate\Queue\Connectors\ConnectorInterface;

class EventBridgeSqsFifoQueueServiceProvider extends ServiceProvider
{
    /**
     * Register any application services.
     *
     * @return void
     */
    public function register()
    {
        $app = $this->app;

        $this->registerDeduplicators();

        // Queue is a deferred provider. We don't want to force resolution to provide
        // a new driver. Therefore, if the queue has already been resolved, extend
        // it now. Otherwise, extend the queue after it has been resolved.
        if ($app->bound('queue')) {
            $this->extendManager($app['queue']);
        } else {
            $app->afterResolving('queue', function ($manager) {
                $this->extendManager($manager);
            });
        }
    }

    /**
     * Register everything for the given manager.
     *
     * @param  \Illuminate\Queue\QueueManager  $manager
     *
     * @return void
     */
    public function extendManager($manager)
    {
        $this->registerConnectors($manager);
    }

    /**
     * Register the connectors on the queue manager.
     *
     * @param  \Illuminate\Queue\QueueManager  $manager
     *
     * @return void
     */
    public function registerConnectors($manager)
    {
        $manager->extend('eventbridge-publisher', function () {
            return new EventBridgeConnector();
        });

        $manager->extend('sqs-consumer', function () {
            return new SqsConsumerConnector();
        });

        $manager->extend('sqs-fifo', function () {
            return new SqsFifoConnector();
        });
    }

    /**
     * Register the default deduplicator methods.
     *
     * @return void
     */
    public function registerDeduplicators()
    {
        foreach (['Unique', 'Content', 'Sqs'] as $deduplicator) {
            $this->{"register{$deduplicator}Deduplicator"}();
        }
    }

    /**
     * Register the unique deduplicator to treat all messages as unique.
     *
     * @return void
     */
    public function registerUniqueDeduplicator()
    {
        $this->app->bind('queue.sqs-fifo.deduplicator.unique', Unique::class);
    }

    /**
     * Register the content deduplicator to treat messages with the same payload as duplicates.
     *
     * @return void
     */
    public function registerContentDeduplicator()
    {
        $this->app->bind('queue.sqs-fifo.deduplicator.content', Content::class);
    }

    /**
     * Register the SQS deduplicator for queues with ContentBasedDeduplication enabled on SQS.
     *
     * @return void
     */
    public function registerSqsDeduplicator()
    {
        $this->app->bind('queue.sqs-fifo.deduplicator.sqs', Sqs::class);
    }
}
