<?php

namespace MoveOn\EventBridgeSqsFifoQueue\Queue;

use Aws\Sqs\SqsClient;
use Aws\EventBridge\EventBridgeClient;
use Carbon\CarbonTimeZone;
use Illuminate\Contracts\Encryption\Encrypter;
use Illuminate\Contracts\Queue\ClearableQueue;
use Illuminate\Contracts\Queue\Queue as QueueContract;
use Illuminate\Queue\CallQueuedClosure;
use Illuminate\Queue\InvalidPayloadException;
use Illuminate\Queue\Jobs\SqsJob;
use Illuminate\Support\Carbon;
use Illuminate\Support\Str;
use Illuminate\Support\Arr;
use Illuminate\Queue\Queue as AbstractQueue;
use MoveOn\EventBridgeSqsFifoQueue\Jobs\BaseMessageBrokerJob;
use MoveOn\EventBridgeSqsFifoQueue\Jobs\MessageBrokerJob;

class EventBridgePublisher extends AbstractQueue implements QueueContract, ClearableQueue
{
    /**
     * The Amazon SQS instance.
     *
     * @var \Aws\EventBridge\EventBridgeClient;
     */
    protected $eventBridge;

    /**
     * The name of the default queue.
     *
     * @var string
     */
    protected $default;

    /**
     * The queue URL prefix.
     *
     * @var string
     */
    protected $prefix;

    /**
     * The queue name suffix.
     *
     * This property was made protected in Laravel 10x. The redefinition
     * here can be removed when support for < Laravel 10x is dropped.
     *
     * @var string
     */
    protected $suffix;

    /**
     * The message group id of the fifo pipe in the queue.
     *
     * @var string
     */
    protected $group;

    /**
     * The driver to generate the deduplication id for the message.
     *
     * @var string
     */
    protected $deduplicator;

    /**
     * The flag to check if this queue is setup for delay.
     *
     * @var bool
     */
    protected $allowDelay;

    /**
     * Create a new Amazon Event bridge queue instance.
     *
     * @param  \Aws\EventBridge\EventBridgeClient  $eventBridge
     * @param  string  $eventBus
     * @param  bool  $dispatchAfterCommit
     * @return void
     */
    public function __construct(EventBridgeClient $eventBridge,
                                $eventBus,
                                $globalSource,
                                $allowDelay = false)
    {
        $this->eventBridge  = $eventBridge;
        $this->globalSource = $globalSource;
        $this->eventBus     = $eventBus;
        $this->allowDelay   = $allowDelay;
    }

    /**
     * Get the size of the queue.
     *
     * @param  string|null  $queue
     * @return int
     */
    public function size($queue = null)
    {
        throw new \Exception("Not implemented");
    }


    /**
     * Push a new job onto the queue.
     *
     * @param  string  $job
     * @param  mixed  $data
     * @param  string|null  $queue
     * @return mixed
     */
    public function push($job, $data = '', $queue = null)
    {
        return $this->enqueueUsing(
            $job,
            $this->createPayload($job, $queue ?: $this->default, $data),
            $queue,
            null,
            function ($payload, $queue) {
                return $this->pushRaw($payload, $queue);
            }
        );
    }

    /**
     * Create a payload string from the given job and data.
     *
     * @param  \Closure|string|object  $job
     * @param  string  $queue
     * @param  mixed  $data
     * @return string
     *
     * @throws \Illuminate\Queue\InvalidPayloadException
     */
    protected function createPayload($job, $queue, $data = '')
    {
        if ($job instanceof Closure) {
            $job = CallQueuedClosure::create($job);
        }

        $payload = json_encode($value = $this->createPayloadArray($job, $queue, $data), \JSON_UNESCAPED_UNICODE);

        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new InvalidPayloadException(
                'Unable to JSON encode payload. Error ('.json_last_error().'): '.json_last_error_msg(), $value
            );
        }

        return $payload;
    }


    /**
     * Push a raw payload onto the queue.
     *
     * @param  string  $payload
     * @param  string|null  $queue
     * @param  array  $options
     * @return mixed
     */
    public function pushRaw($payload, $queue = null, array $options = [])
    {
        $finalPayload = $payload;
//        if (($deduplication = $this->getDeduplicationId($payload, $queue)) !== false) {
//            $finalPayload['MessageDeduplicationId'] = $deduplication;
//        }
        $entry = [
            'Time'          => Carbon::now()->timestamp,
            'Source'        => $this->globalSource,
            'Resources'     => [],
            'DetailType'    => "private",
            'Detail'        => $finalPayload,
            'EventBusName'  => $this->eventBus
        ];
        $response = $this->eventBridge->putEvents(
            [
                'Entries' => [$entry]
            ]
        )->get('Entries');
        return $response["EventId"] ?? null;
    }

    /**
     * Create a payload array from the given job and data.
     *
     * @param  string|object  $job
     * @param  string  $queue
     * @param  mixed  $data
     * @return array
     */
    protected function createPayloadArray($job, $queue, $data = '')
    {
        return is_object($job)
            ? $this->createObjectPayload($job, $queue)
            : $this->createStringPayload($job, $queue, $data);
    }

    /**
     * Create a payload for an object-based queue handler.
     *
     * @param  object  $job
     * @param  string  $queue
     * @return array
     */
    protected function createObjectPayload($job, $queue)
    {
        $payload = $this->withCreatePayloadHooks($queue, [
            "sourceSystem" => $this->getSourceSystem($job),
            "component" => $this->getComponent($job),
            "action" => $this->getAction($job),
            "entity" => $this->getEntity($job),
            "entityId" => $this->getEntityId($job),
            'uuid' => (string) Str::uuid(),
            'displayName' => $this->getDisplayName($job),
            'job' => 'Illuminate\Queue\CallQueuedHandler@call',
            'maxTries' => $job->tries ?? null,
            'maxExceptions' => $job->maxExceptions ?? null,
            'failOnTimeout' => $job->failOnTimeout ?? false,
            'backoff' => $this->getJobBackoff($job),
            'timeout' => $job->timeout ?? null,
            'retryUntil' => $this->getJobExpiration($job),
            'data' => [
                'commandName' => $job,
                'command' => $job,
            ],
        ]);

        $command = $this->jobShouldBeEncrypted($job) && $this->container->bound(Encrypter::class)
            ? $this->container[Encrypter::class]->encrypt(serialize(clone $job))
            : serialize(clone $job);

        return array_merge($payload, [
            'data' => array_merge($payload['data'], [
                'commandName' => get_class($job),
                'command' => $command,
            ]),
        ]);
    }

    /**
     * Create a typical, string based queue payload array.
     *
     * @param  string  $job
     * @param  string  $queue
     * @param  mixed  $data
     * @return array
     */
    protected function createStringPayload($job, $queue, $data)
    {
        return $this->withCreatePayloadHooks($queue, [
            "sourceSystem" => $this->getSourceSystem($job),
            "component" => $this->getComponent($job),
            "action" => $this->getAction($job),
            "entity" => $this->getEntity($job),
            "entityId" => $this->getEntityId($job),
            'uuid' => (string) Str::uuid(),
            'displayName' => is_string($job) ? explode('@', $job)[0] : null,
            'job' => $job,
            'maxTries' => null,
            'maxExceptions' => null,
            'failOnTimeout' => false,
            'backoff' => null,
            'timeout' => null,
            'data' => $data,
        ]);
    }

    /**
     * Get the backoff for an object-based queue handler.
     *
     * @param  mixed  $job
     * @return mixed
     */
    public function getSourceSystem($job)
    {
        if (is_subclass_of($job, BaseMessageBrokerJob::class)) {
            return $job->sourceSystem;
        }
        if (!property_exists($job, "data") || count($job->data) === 0) {
            return;
        }
        $event = $job->data[0];
        if (! method_exists($event, 'sourceSystem') && ! isset($event->sourceSystem)) {
            return;
        }

        if (is_null($sourceSystem = $event->sourceSystem ?? $event->sourceSystem())) {
            return;
        }
        return $sourceSystem;
    }

    /**
     * Get the backoff for an object-based queue handler.
     *
     * @param  mixed  $job
     * @return mixed
     */
    public function getAction($job)
    {
        if (is_subclass_of($job, BaseMessageBrokerJob::class)) {
            return $job->action;
        }

        if (!property_exists($job, "data") ||count($job->data) === 0) {
            return;
        }
        $event = $job->data[0];
        if (! method_exists($event, 'action') && ! isset($event->action)) {
            return;
        }

        if (is_null($action = $event->action ?? $event->action())) {
            return;
        }

        return $action;
    }


    /**
     * Get the backoff for an object-based queue handler.
     *
     * @param  mixed  $job
     * @return mixed
     */
    public function getComponent($job)
    {
        if (is_subclass_of($job, BaseMessageBrokerJob::class)) {
            return $job->component;
        }

        if (!property_exists($job, "data") ||count($job->data) === 0) {
            return;
        }
        $event = $job->data[0];

        if (! method_exists($event, 'component') && ! isset($event->component)) {
            return;
        }

        if (is_null($component = $event->component ?? $event->component())) {
            return;
        }

        return $component;
    }

    /**
     * Get the backoff for an object-based queue handler.
     *
     * @param  mixed  $job
     * @return mixed
     */
    public function getEntity($job)
    {
        if (is_subclass_of($job, BaseMessageBrokerJob::class)) {
            return $job->entity;
        }

        if (!property_exists($job, "data") || count($job->data) === 0) {
            return;
        }
        $event = $job->data[0];

        if (! method_exists($event, 'entity') && ! isset($event->entity)) {
            return;
        }

        if (is_null($entity = $event->entity ?? $event->entity())) {
            return;
        }

        return $entity;
    }

    /**
     * Get the backoff for an object-based queue handler.
     *
     * @param  mixed  $job
     * @return mixed
     */
    public function getEntityId($job)
    {
        if (is_subclass_of($job, BaseMessageBrokerJob::class)) {
            return $job->entityId;
        }

        if (!property_exists($job, "data") || count($job->data) === 0) {
            return;
        }
        $event = $job->data[0];

        if (! method_exists($event, 'entityId') && ! isset($event->entityId)) {
            return;
        }

        if (is_null($entityId = $event->entityId ?? $event->entityId())) {
            return;
        }

        return $entityId;
    }

    /**
     * Push a new job onto the queue after (n) seconds.
     *
     * @param  \DateTimeInterface|\DateInterval|int  $delay
     * @param  string  $job
     * @param  mixed  $data
     * @param  string|null  $queue
     * @return mixed
     */
    public function later($delay, $job, $data = '', $queue = null)
    {
        return $this->enqueueUsing(
            $job,
            $this->createPayload($job, $queue ?: $this->default, $data),
            $queue,
            $delay,
            function ($payload, $queue, $delay) {
                $this->pushRaw($payload, $queue);
            }
        );
    }

    /**
     * Push an array of jobs onto the queue.
     *
     * @param  array  $jobs
     * @param  mixed  $data
     * @param  string|null  $queue
     * @return void
     */
    public function bulk($jobs, $data = '', $queue = null)
    {
        foreach ((array) $jobs as $job) {
            if (isset($job->delay)) {
                $this->later($job->delay, $job, $data, $queue);
            } else {
                $this->push($job, $data, $queue);
            }
        }
    }

    /**
     * Pop the next job off of the queue.
     *
     * @param  string|null  $queue
     * @return \Illuminate\Contracts\Queue\Job|null
     */
    public function pop($queue = null)
    {
      throw new \Exception("Not implemented");
    }

    /**
     * Delete all of the jobs from the queue.
     *
     * @param  string  $queue
     * @return int
     */
    public function clear($queue)
    {
        throw new \Exception("Not implemented");
    }

}
