<?php

namespace MoveOn\Inventory\Client;

use Exception;
use Illuminate\Http\Client\PendingRequest;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use MoveOn\Inventory\Client\Exception\InventoryRequestFailed;
use MoveOn\Inventory\Client\Exception\ProductNotFoundException;

class Inventory
{
    /**
     * @var string
     */
    public static string $version = "2.0.0";

    /**
     * @var PendingRequest
     */
    private PendingRequest $apiClient;

    /**
     * The inventory constructor.
     *
     * @param array $config [description]
     */
    public function __construct()
    {
        $this->apiClient = Http::withHeaders([
            "Accept" => "application/json",
        ])->baseUrl(
            config("services.inventory.inventory_host")
        );
    }

    /**
     * Search to inventory
     *
     * @throws InventoryRequestFailed
     */
    public function search(
        array $body=[], string $response_format = "overview",
        string $version = "1", int $page = 1, int $perPage = 20)
    {
        $query = http_build_query([
            "version"         => $version,
            "page"            => $page,
            "per_page"        => $perPage,
            "response_format" => $response_format
        ]);

        $response = $this->apiClient->post(
            "api/product-search/product/v1/products?$query",
            [
                "body" => $body
            ]
        );

        if ($response->failed()) {
            Log::error("Product API Server Error. Response Code", [
                "error" => $response->json()
            ]);
            throw new InventoryRequestFailed(
                "Product API Server Error. Response Code: " . $response->getStatusCode()
            );
        }

        return $response->json();
    }

    /**
     * Find product by ID
     *
     * @throws InventoryRequestFailed|Exception
     */
    public function byId(string $id)
    {
        $response = $this->apiClient->get("api/product-search/product/v1/products/$id");

        if ($response->failed()) {
            throw match ($response->status()) {
                404 => new ProductNotFoundException(),
                default => new InventoryRequestFailed(
                    "Product API Server Error. Response Code: ".$response->status()
                ),
            };
        }

        return $response->json();
    }

    /**
     * Find product by URL
     *
     * @throws InventoryRequestFailed|Exception
     */
    public function byUrl(string $url)
    {
        if (!filter_var($url, FILTER_VALIDATE_URL)
            || !str_starts_with($url, "http")) {
            throw new Exception("The URL is invalid.");
        }

        $response = $this->apiClient->get("api/product-search/product/v1/products", [
            "url" => $url
        ]);


        if ($response->failed()) {
            throw match ($response->status()) {
                404 => new ProductNotFoundException(),
                default => new InventoryRequestFailed(
                    "Product API Server Error. Response Code: ".$response->status()
                ),
            };
        }

        return $response->json();
    }


    /**
     * Find product by ID
     *
     * @throws InventoryRequestFailed
     */
    public function getDescriptionById(string $id)
    {
        $response = $this->apiClient->get("api/product-search/product/v1/products/$id/description");

        if ($response->failed()) {
            throw new InventoryRequestFailed(
                "Product API Server Error. Response Code: " . $response->status()
            );
        }

        return $response->json();
    }

    /**
     * Find product by ID
     *
     * @throws InventoryRequestFailed
     */
    public function getReviewsById(string $id)
    {
        $response = $this->apiClient->get("api/product-search/product/v1/products/$id/reviews");

        if ($response->failed()) {
            throw new InventoryRequestFailed(
                "Product API Server Error. Response Code: " . $response->status()
            );
        }

        return $response->json();
    }
}
