<?php

namespace MoveOn\Inventory\Client\Models;

use MoveOn\Inventory\Client\Collection\Attributes;
use MoveOn\Inventory\Client\Collection\Categories;
use MoveOn\Inventory\Client\Collection\Gallery;
use MoveOn\Inventory\Client\Collection\Skus;
use MoveOn\Inventory\Client\Collection\Specifications;

class Product extends Model
{
    /**
     * @var array
     */
    protected array $price;

    /**
     * @var array
     */
    protected array $discountPrice;

    /**
     * @var array
     */
    protected array $ratings = [];

    /**
     * @var mixed|null
     */
    protected $meta = null;

    /**
     * @var \MoveOn\Inventory\Client\Collection\Attributes
     */
    protected Attributes $variations;

    /**
     * @var \MoveOn\Inventory\Client\Collection\Skus
     */
    protected Skus $skus;

    /**
     * @var \MoveOn\Inventory\Client\Collection\Gallery
     */
    protected Gallery $gallery;

    /**
     * @var \MoveOn\Inventory\Client\Models\Source
     */
    protected Source $source;

    /**
     * @var \MoveOn\Inventory\Client\Models\Seller
     */
    protected Seller $seller;

    /**
     * @var \MoveOn\Inventory\Client\Collection\Categories
     */
    protected Categories $categories;

    /**
     * @var \MoveOn\Inventory\Client\Collection\Specifications
     */
    protected Specifications $specifications;

    /**
     * Product constructor.
     *
     * @throws \Exception
     */
    public function __construct(array $attributes)
    {
        $defaults      = [
            'title'     => '',
            'link'      => '',
            'image'     => '',
            'thumbnail' => '',
            'min_price' => '',
            'max_price' => '',
        ];
        $allowedFields = array_keys($defaults);
        $cleanedAttrs  = array_filter($attributes, fn($attr) => in_array($attr, $allowedFields) && (is_string($attr) || is_numeric('')));
        $diffAttr      = array_diff_key($cleanedAttrs, $defaults);

        if (count($diffAttr)) {
            return throw new \Exception('Not enough arguments. Please fill require attributes: ' . implode(', ', array_keys($diffAttr)));
        }

        $this->attributes    = $attributes;
        $this->price         = [
            'min' => $attributes['min_price'],
            'max' => $attributes['max_price'],
        ];
        $minDiscount         = $attributes['min_discount_price'] ?? null;
        $this->discountPrice = [
            'min' => $minDiscount,
            'max' => $attributes['max_discount_price'] ?? $minDiscount,
        ];

        $this->meta = $attributes['meta'] ?? null;

        $this->gallery        = new Gallery();
        $this->variations     = new Attributes();
        $this->skus           = new Skus();
        $this->source         = new Source();
        $this->categories     = new Categories();
        $this->seller         = new Seller();
        $this->specifications = new Specifications();
    }

    /**
     * Get the product title
     *
     * @return string|null
     */
    public function title(): ?string
    {
        return $this->attributes['title'] ?? null;
    }

    /**
     * The product link.
     *
     * @return string|null
     */
    public function link(): ?string
    {
        return $this->attributes['link'] ?? null;
    }

    /**
     * Get the product image.
     *
     * @return string|null
     */
    public function image(): ?string
    {
        return $this->attributes['image'] ?? null;
    }

    /**
     * Get the product thumbnail.
     *
     * @return string|null
     */
    public function thumbnail(): ?string
    {
        return $this->attributes['thumbnail'] ?? null;
    }

    /**
     * Get the product price.
     *
     * @return array
     */
    public function price(): array
    {
        return $this->price;
    }

    /**
     * Get the discount price.
     *
     * @return array
     */
    public function discountPrice(): array
    {
        return $this->discountPrice;
    }

    /**
     * Get the wholesale price.
     *
     * @return array
     */
    public function wholesalePrice(): array
    {
        return $this->attributes['wholesales'] ?? [];
    }

    /**
     * Get the gallery.
     *
     * @return \MoveOn\Inventory\Client\Collection\Gallery
     */
    public function gallery(): Gallery
    {
        return $this->gallery;
    }

    /**
     * Get the product attributes.
     *
     * @return \MoveOn\Inventory\Client\Collection\Attributes
     */
    public function attributes(): Attributes
    {
        return $this->variations;
    }

    /**
     * Get the product Skus.
     *
     * @return \MoveOn\Inventory\Client\Collection\Skus
     */
    public function skus(): Skus
    {
        return $this->skus;
    }

    /**
     * Get the product source.
     *
     * @return \MoveOn\Inventory\Client\Models\Source
     */
    public function source(): Source
    {
        return $this->source;
    }

    /**
     * Set source.
     *
     * @param \MoveOn\Inventory\Client\Models\Source $source
     */
    public function setSource(Source $source)
    {
        $this->source = $source;
    }

    /**
     * Get the product seller.
     *
     * @return \MoveOn\Inventory\Client\Models\Seller
     */
    public function seller(): Seller
    {
        return $this->seller;
    }

    /**
     * Set seller.
     *
     * @param \MoveOn\Inventory\Client\Models\Seller $seller
     */
    public function setSeller(Seller $seller)
    {
        $this->seller = $seller;
    }

    /**
     * Get the product categories.
     *
     * @return \MoveOn\Inventory\Client\Collection\Categories
     */
    public function categories(): Categories
    {
        return $this->categories;
    }

    /**
     * Get the product specification.
     *
     * @return \MoveOn\Inventory\Client\Collection\Specifications
     */
    public function specifications(): Specifications
    {
        return $this->specifications;
    }

    /**
     * Set five-star ratings.
     *
     * @param int $one
     * @param int $two
     * @param int $three
     * @param int $four
     * @param int $five
     */
    public function setRatings(int $one, int $two, int $three, int $four, int $five)
    {
        $this->ratings = [
            $one, $two, $three, $four, $five,
        ];
    }

    /**
     * Get five-star ratings.
     *
     * @return array
     */
    public function ratings(): array
    {
        return $this->ratings;
    }

    /**
     * Return the product as array.
     *
     * @return array
     */
    public function toArray(): array
    {
        return [
            'id'             => $this->id,
            'vpid'           => $this->vpid,
            'vendor'         => $this->vendor,
            'title'          => $this->title(),
            'link'           => $this->link(),
            'image'          => $this->image(),
            'thumbnail'      => $this->thumbnail(),
            'sold'           => $this->sold,
            'stock'          => $this->stock,
            'ratings'        => $this->ratings(),
            'rating'         => $this->rating,
            'rating_count'   => $this->rating_count,
            'status'         => null,
            'countdown'      => null,
            'badge'          => null,
            'price'          => $this->price(),
            'meta'           => $this->meta,
            'discount_price' => $this->discountPrice(),
            'wholesales'     => $this->wholesalePrice(),
            'shipping_cost'  => $this->shipping_cost,
            'gallery'        => $this->gallery()->toArray(),
            'skus'           => $this->skus()->toArray(),
            'attributes'     => $this->attributes()->toArray(),
            'source'         => $this->source()->toArray(),
            'seller'         => $this->seller()->toArray(),
            'categories'     => $this->categories()->toArray(),
            'specifications' => $this->specifications()->toArray(),
        ];
    }
}
