<?php

namespace MoveOn\Inventory\Client;

use Exception;
use GuzzleHttp\Client;
use GuzzleHttp\Exception\GuzzleException;
use MoveOn\Inventory\Client\Collection\Freights;
use MoveOn\Inventory\Client\Collection\Products;
use MoveOn\Inventory\Client\Helpers\CategoryTree;
use MoveOn\Inventory\Client\Models\Attribute;
use MoveOn\Inventory\Client\Models\Freight;
use MoveOn\Inventory\Client\Models\Media;
use MoveOn\Inventory\Client\Models\Product;
use MoveOn\Inventory\Client\Models\Seller;
use MoveOn\Inventory\Client\Models\Sku;
use MoveOn\Inventory\Client\Models\Source;
use MoveOn\Inventory\Client\Models\Specification;
use Psr\Http\Message\ResponseInterface;

class Inventory
{
    /**
     * @var string
     */
    public static string $version = '1.0.0';

    /**
     * @var string|null
     */
    protected ?string $appUrl;

    /**
     * @var string|null
     */
    protected ?string $appId;

    /**
     * @var string|null
     */
    protected ?string $appSecret;

    /**
     * @var \GuzzleHttp\Client
     */
    private Client $apiClient;

    /**
     * The inventory constructor.
     *
     * @param array $config [description]
     */
    public function __construct(array $config)
    {
        $this->appUrl    = $config['app_url'] ?? null;
        $this->appId     = $config['app_id'] ?? null;
        $this->appSecret = $config['app_secret'] ?? null;

        $this->apiClient = new Client([
            'base_uri'    => $this->appUrl,
            'http_errors' => false,
            'connect_timeout' => $config['connect_timeout'] ?? 5,
            'timeout' => $config['timeout'] ?? 10,
            'headers'     => [
                'X-API-CLIENT'  => self::$version,
                'X-API-APP-ID'  => $this->appId,
                'Authorization' => 'Bearer ' . $this->appSecret,
            ],
        ]);
    }

    /**
     * Search in the inventory.
     *
     * @param array $filters
     *
     * @return \MoveOn\Inventory\Client\Collection\Products
     * @throws \Exception
     */
    public function search(array $filters): Products
    {
        try {
            $request = $this->apiClient->get('/api/v1/products', [
                'query' => $filters,
            ]);

            if ($request->getStatusCode() !== 200) {
                throw new Exception('API Server Error. Response Code: ' . $request->getStatusCode());
            }

            $body     = $request->getBody()->getContents();
            $results  = json_decode($body, true);
            $products = $results['data'] ?? [];

            $collection = new Products();

            foreach ($products as $product) {
                $model = new Product([
                    'id'                 => $product['id'],
                    'vpid'               => $product['vpid'],
                    'vendor'             => $product['vendor'],
                    'title'              => $product['title'],
                    'link'               => $product['link'],
                    'image'              => $product['image'],
                    'thumbnail'          => $product['image'],
                    'min_price'          => $product['price_real'] ?? $product['price'],
                    'max_price'          => $product['price_real'] ?? $product['price'],
                    'min_discount_price' => $product['price_real'] ? $product['price'] : null,
                    'max_discount_price' => $product['price_real'] ? $product['price'] : null,
                    'sold'               => $product['orders'],
                    'rating'             => $product['rating'],
                    'rating_count'       => $product['rating_count'],
                    'shipping_cost'      => $product['shipping_cost'],
                ]);

                $collection->addProduct($model);
            }
            $collection->setPaginate(
                $results['total'] ?? 0,
                $results['per_page'] ?? 0,
                $results['current_page'] ?? 0
            );

            $seller  = $results['seller'] ?? null;
            $filters = $results['filters'] ?? null;

            if ($seller) {
                $collection->setExtras('seller', $seller);
            }

            if ($filters) {
                $collection->setExtras('filters', $filters);
            }

            return $collection;
        } catch (GuzzleException $exception) {
            throw new Exception('Guzzle Error: ' . $exception->getMessage());
        }
    }

    /**
     * Get product by link.
     *
     * @param string $link
     * @param bool $refresh
     *
     * @return \MoveOn\Inventory\Client\Models\Product
     * @throws \Exception
     */
    public function byLink(string $link, bool $refresh = false): Product
    {
        if (!filter_var($link, FILTER_VALIDATE_URL) || strpos($link, 'http') !== 0) {
            throw new Exception('The link is invalid.');
        }

        try {
            $request = $this->apiClient->post('/api/v1/products/fetch', [
                'form_params' => [
                    'url'     => $link,
                    'refresh' => $refresh ? 'yes' : 'no',
                ],
            ]);

            return $this->prepareProductDetails($request);
        } catch (GuzzleException $exception) {
            throw new Exception('Guzzle Error: ' . $exception->getMessage());
        }
    }

    /**
     * Get product by inventory id.
     *
     * @param string $id
     * @param bool $refresh
     *
     * @return \MoveOn\Inventory\Client\Models\Product
     * @throws \Exception
     */
    public function byId(string $id, bool $refresh = false): Product
    {
        try {
            $request = $this->apiClient->get('/api/v1/products/details', [
                'query' => [
                    'id'      => $id,
                    'refresh' => $refresh ? 'yes' : 'no',
                ],
            ]);

            return $this->prepareProductDetails($request);
        } catch (GuzzleException $exception) {
            throw new Exception('Guzzle Error: ' . $exception->getMessage());
        }
    }

    /**
     * Get product vendor product id.
     *
     * @param string $vendor
     * @param string $vPid
     * @param bool $refresh
     *
     * @return \MoveOn\Inventory\Client\Models\Product
     * @throws \Exception
     */
    public function byVendorId(string $vendor, string $vPid, bool $refresh = false): Product
    {
        try {
            $request = $this->apiClient->get('/api/v1/products/fetch-by-id', [
                'query' => [
                    'shop'    => $vendor,
                    'id'      => $vPid,
                    'refresh' => $refresh ? 'yes' : 'no',
                ],
            ]);

            return $this->prepareProductDetails($request);
        } catch (GuzzleException $exception) {
            throw new Exception('Guzzle Error: ' . $exception->getMessage());
        }
    }

    /**
     * Fetch reviews.
     *
     * @param string $id
     * @param int $page
     *
     * @return array
     * @throws \Exception
     */
    public function fetchReviews(string $id, int $page = 1): array
    {
        try {
            $request = $this->apiClient->get('/api/v1/content', [
                'query' => [
                    'id'     => $id,
                    'type'   => 'reviews',
                    'page'   => $page,
                    'markup' => 'no',
                ],
            ]);

            $data = (array) json_decode($request->getBody()->getContents(), true);

            return $data['data'] ?? [];
        } catch (GuzzleException $exception) {
            throw new Exception('Guzzle Error: ' . $exception->getMessage());
        }
    }

    /**
     * Fetch body content.
     *
     * @param string $id
     *
     * @return string|null
     * @throws \Exception
     */
    public function fetchBody(string $id): ?string
    {
        try {
            $request = $this->apiClient->get('/api/v1/content', [
                'query' => [
                    'id'     => $id,
                    'type'   => 'body',
                    'markup' => 'yes',
                ],
            ]);

            if ($request->getStatusCode() === 200) {
                return $request->getBody()->getContents();
            }
        } catch (GuzzleException $exception) {
            throw new Exception('Guzzle Error: ' . $exception->getMessage());
        }

        return null;
    }

    /**
     * Prepare product details.
     *
     * @param \Psr\Http\Message\ResponseInterface $request
     *
     * @return \MoveOn\Inventory\Client\Models\Product
     * @throws \Exception
     */
    private function prepareProductDetails(ResponseInterface $request): Product
    {
        if ($request->getStatusCode() !== 200) {
            throw new Exception('Product API Server Error. Response Code: ' . $request->getStatusCode());
        }

        $body    = $request->getBody()->getContents();
        $results = json_decode($body, true);
        $status  = $results['status'] ?? 'error';

        if ($status !== 'success') {
            throw new Exception('Product Details API Server Error: ' . ($results['message'] ?? 'Unknown error.'));
        }

        return $this->makeProduct($results['data']);
    }

    /**
     * Make the product.
     *
     * @param array $data
     *
     * @return \MoveOn\Inventory\Client\Models\Product
     * @throws \Exception
     */
    private function makeProduct(array $data): Product
    {
        $product = new Product([
            'id'                 => $data['id'],
            'vpid'               => $data['vid'],
            'vendor'             => $data['meta']['vendor'] ?? null,
            'shop_id'            => $data['shop_id'] ?? null,
            'seller_id'          => $data['seller_id'] ?? null,
            'title'              => $data['title'],
            'link'               => $data['link'],
            'image'              => $data['image'],
            'thumbnail'          => $data['image'],
            'min_price'          => $data['price']['original']['min'],
            'max_price'          => $data['price']['original']['max'],
            'min_discount_price' => $data['price']['discount']['min'],
            'max_discount_price' => $data['price']['discount']['max'],
            'wholesales'         => $data['price']['wholesales'] ?? [],
            'sold'               => $data['sales'],
            'stock'              => $data['stock'],
            'rating'             => $data['ratings_average'],
            'rating_count'       => $data['ratings_count'],
            'shipping_cost'      => $data['shipping_cost'] ?? null,
            'weight'             => $data['meta']['weight'] ?? null,
            'weight_type'        => $data['meta']['weight_type'] ?? null,
            'meta'               => $data['meta'] ?? null,
        ]);

        // Set Ratting.
        if (count((array) $data['ratings']) === 5) {
            [$one, $two, $three, $four, $five] = $data['ratings'];
            $product->setRatings($one, $two, $three, $four, $five);
        }

        // Set Media
        $videos = (array) ($data['meta']['videos'] ?? []);
        foreach ($videos as $video) {
            $product->gallery()->addMedia(new Media(
                $video['preview'],
                $video['preview'],
                null,
                $video['url'],
            ));
        }

        foreach ($data['gallery'] as $media) {
            $product->gallery()->addMedia(new Media(
                $media['url'],
                $media['thumb'],
                $media['title'],
            ));
        }

        // Set SKUs
        $skus = (array) ($data['variation']['skus'] ?? []);
        foreach ($skus as $sku) {
            $skuModel = new Sku(
                $sku['id'],
                $sku['props'],
                $sku['price']['actual'],
                $sku['stock']['available']
            );
            $skuModel->setAttribute('discount_price', $sku['price']['offer'] ?? null);
            $skuModel->setAttribute('special_price', $sku['price']['preorder'] ?? null);
            $skuModel->setAttribute('min_qty', $sku['stock']['min'] ?? null);
            $skuModel->setAttribute('limit_qty', $sku['stock']['limit'] ?? null);

            $product->skus()->addSku($skuModel);
        }

        // Set attributes
        $attributes = (array) ($data['variation']['props'] ?? []);
        foreach ($attributes as $attribute) {
            $variation = new Attribute($attribute['id'], $attribute['name']);
            $values    = (array) ($attribute['values'] ?? []);

            foreach ($values as $value) {
                $variation->setValue(
                    $value['id'],
                    $value['title'],
                    $value['image'] ?? null,
                    $value['thumb'] ?? null,
                    $value['color'] ?? null,
                    $value['name'] ?? null,
                );
            }

            $product->attributes()->addAttribute($variation);
        }

        // Set Source
        $source = $data['shop'] ?? [];
        $product->setSource(new Source([
            'id'         => $source['id'] ?? null,
            'name'       => $source['name'] ?? null,
            'url'        => $source['url'] ?? null,
            'identifier' => $source['identifier'] ?? null,
            'country'    => $source['country_code'] ?? null,
            'currency'   => $source['currency_code'] ?? null,
        ]));

        // Set Seller
        $seller = $data['seller'] ?? [];
        $product->setSeller(new Seller([
            'id'          => $seller['id'] ?? null,
            'name'        => $seller['name'] ?? null,
            'description' => $seller['description'] ?? null,
            'url'         => $seller['link'] ?? null,
            'source'      => $seller['vendor'] ?? null,
            'meta'        => $seller['meta'] ?? null,
            'source_id'   => $seller['vendor_id'] ?? null,
            'opened_at'   => $seller['opened_at'] ?? null,
        ]));

        // Set Categories
        $categories   = $data['categories'] ?? [];
        $categoryTree = CategoryTree::make($categories)->get();

        if (count($categoryTree) > 0) {
            foreach ($categoryTree as $category) {
                $product->categories()->addCategory($category);
            }
        }

        // Set Specifications
        $specifications = (array) ($data['specifications'] ?? []);
        foreach ($specifications as $specification) {
            $product->specifications()->addSpecification(new Specification(
                $specification['label']['name'],
                $specification['value']['name'],
                $specification['label']['id'],
                $specification['value']['id'],
            ));
        }

        return $product;
    }

    /**
     * Fetch freights for a product.
     *
     * @param int $id
     * @param int $quantity
     * @param string|null $skuId
     *
     * @return \MoveOn\Inventory\Client\Collection\Freights
     * @throws \Exception
     */
    public function fetchFreights(int $id, int $quantity, ?string $skuId = null): Freights
    {
        try {
            $request = $this->apiClient->get('/api/v1/shipping', [
                'query' => [
                    'pid' => $id,
                    'qty' => $quantity,
                    'sku' => $skuId,
                ],
            ]);

            if ($request->getStatusCode() !== 200) {
                throw new Exception('Shipping API Server Error. Response Code: ' . $request->getStatusCode());
            }

            $data       = json_decode($request->getBody()->getContents(), true);
            $methods    = $data['methods'] ?? [];
            $collection = new Freights();

            foreach ($methods as $method) {
                $collection->addFreight(new Freight($method));
            }

            return $collection;
        } catch (GuzzleException $exception) {
            throw new Exception('Guzzle Error: ' . $exception->getMessage());
        }
    }

    /**
     * Fetch domestic fee for a product.
     *
     * @param int $id
     * @param int $quantity
     *
     * @return mixed
     * @throws \Exception
     */
    public function fetchDomesticFee(int $id, int $quantity)
    {
        try {
            $request = $this->apiClient->get('/api/v1/domestic-fee', [
                'query' => [
                    'pid' => $id,
                    'qty' => $quantity,
                ],
            ]);

            if ($request->getStatusCode() !== 200) {
                throw new Exception('Shipping API Server Error. Response Code: ' . $request->getStatusCode());
            }

            return json_decode($request->getBody()->getContents(), true);
        } catch (GuzzleException $exception) {
            throw new Exception('Guzzle Error: ' . $exception->getMessage());
        }
    }
}
