<?php

namespace MoveOn\Inventory\Client\Collection;

use MoveOn\Inventory\Client\Models\Product;

class Products extends Collection
{
    /**
     * @var array
     */
    protected array $items = [];

    /**
     * @var array
     */
    protected array $extras = [];

    /**
     * @var array
     */
    protected array $paginate = [
        'per_page'     => null,
        'total'        => null,
        'current_page' => null,
        'last_page'    => null,
        'prev_page'    => null,
        'next_page'    => null,
        'from'         => null,
        'to'           => null,
    ];

    /**
     * Add a product to the collection.
     *
     * @param \MoveOn\Inventory\Client\Models\Product $product
     *
     * @return self
     */
    public function addProduct(Product $product): self
    {
        $this->add($product);

        return $this;
    }

    /**
     * Set paginate information.
     *
     * @param int $totalItems
     * @param int $perPage
     * @param int $currentPage
     *
     * @return self
     */
    public function setPaginate(int $totalItems, int $perPage, int $currentPage): self
    {
        $lastPage        = $perPage > 0 ? max((int) ceil($totalItems / $perPage), 1) : $currentPage;
        $firstItemNumber = $totalItems > 0 ? ($currentPage - 1) * $perPage + 1 : null;

        $this->paginate = [
            'per_page'     => $perPage,
            'total'        => $totalItems,
            'current_page' => $currentPage,
            'last_page'    => $lastPage,
            'prev_page'    => $currentPage > 1 ? $currentPage - 1 : null,
            'next_page'    => $currentPage < $lastPage ? $currentPage + 1 : null,
            'from'         => $firstItemNumber,
            'to'           => $totalItems > 0 ? $firstItemNumber + count($this->items) - 1 : null,
        ];

        return $this;
    }

    /**
     * Set extras.
     *
     * @param string $key
     * @param mixed $value
     *
     * @return self
     */
    public function setExtras(string $key, $value): self
    {
        $this->extras[$key] = $value;

        return $this;
    }

    /**
     * Get all products array.
     *
     * @return array
     */
    public function toArray(): array
    {
        $items = parent::toArray();

        $data = [
            'items'    => $items,
            'paginate' => $this->paginate,
        ];

        return array_merge($this->extras, $data);
    }
}
