<?php

namespace MoveOn\MetaField\Tests\Unit;

use MoveOn\MetaField\Enum\MetaFieldDimensionUnitsEnum;
use MoveOn\MetaField\Enum\MetaFieldTypesEnum;
use MoveOn\MetaField\Enum\MetaFieldVolumeUnitsEnum;
use MoveOn\MetaField\Enum\MetaFieldWeightUnitsEnum;
use MoveOn\MetaField\Exception\MetaFieldException;
use MoveOn\MetaField\Models\MetaField;
use MoveOn\MetaField\Facades\MetaField as MetaFieldFacade;
use MoveOn\MetaField\Tests\TestCase;

class MetaFieldServiceTest extends TestCase
{
    public $metaFieldService;

    public function test__construct()
    {
        MetaField::query()->truncate();
        $this->assertTrue(true);
    }

    /**
     * Meta field create multiple test
     */
    public function test_create_multiple_meta_field(): void
    {
        $data = [
            [
                'type'  => MetaFieldTypesEnum::BOOLEAN,
                'key'   => 'boolean_key',
                'value' => true
            ],
            [
                'type'  => MetaFieldTypesEnum::DATE_TIME,
                'key'   => 'date_time_key',
                'value' => now()
            ],
            [
                'type'  => MetaFieldTypesEnum::NUMBER_DECIMAL,
                'key'   => 'number_decimal_key',
                'value' => 543543354.544520
            ],
            [
                'type'  => MetaFieldTypesEnum::NUMBER_INTEGER,
                'key'   => 'number_integer_key',
                'value' => 9007199254740991
            ],
            [
                'type'  => MetaFieldTypesEnum::SINGLE_LINE_TEXT_FIELD,
                'key'   => 'single_line_text_field_key',
                'value' => fake()->jobTitle
            ],
            [
                'type'  => MetaFieldTypesEnum::MULTI_LINE_TEXT_FIELD,
                'key'   => 'multi_line_text_field_key',
                'value' => fake()->text(1000)
            ],
            [
                'type'  => MetaFieldTypesEnum::JSON,
                'key'   => 'json_key',
                'value' => json_encode(["ingredient" => "flour", "amount" => 0.3])
            ],
            [
                'type'  => MetaFieldTypesEnum::DIMENSION,
                'key'   => 'dimension_key',
                'value' => ['value' => 2.5, 'unit' => MetaFieldDimensionUnitsEnum::CM]
            ],
            [
                'type'  => MetaFieldTypesEnum::JSON,
                'key'   => 'json_key',
                'value' => "PHP is not dead"
            ],
        ];

        $metaField = MetaFieldFacade::createMultiple(
            4,
            'App\Models\User',
            $data
        );
        $this->assertNull($metaField);
    }

    /**
     * Meta field create test
     */

    // Boolean tests
    public function test_create_meta_field_boolean_type(): void
    {
        $metaField = MetaFieldFacade::create(
            4,
            'App\Models\User',
            'boolean_key',
            MetaFieldTypesEnum::BOOLEAN,
            true
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_boolean_type_value(): void
    {
        $this->expectException(MetaFieldException::class);
        MetaFieldFacade::create(
            4,
            'App\Models\User',
            'boolean_key',
            MetaFieldTypesEnum::BOOLEAN,
            "instead of boolean"
        );
    }

    // Date time tests
    public function test_create_meta_field_date_time_type(): void
    {
        $metaField = MetaFieldFacade::create(
            5,
            'App\Models\User',
            'date_time_key',
            MetaFieldTypesEnum::DATE_TIME,
            now()
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_date_time_value(): void
    {
        $this->expectException(MetaFieldException::class);
        MetaFieldFacade::create(
            5,
            'App\Models\User',
            'date_time_key',
            MetaFieldTypesEnum::DATE_TIME,
            'invalid date time value'
        );
    }

    // Decimal number tests
    public function test_create_meta_field_number_decimal_type(): void
    {
        $metaField = MetaFieldFacade::create(
            6,
            'App\Models\User',
            'number_decimal_key',
            MetaFieldTypesEnum::NUMBER_DECIMAL,
            543543354.544520
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_number_decimal_value(): void
    {
        $this->expectException(MetaFieldException::class);
        MetaFieldFacade::create(
            6,
            'App\Models\User',
            'number_decimal_key',
            MetaFieldTypesEnum::NUMBER_DECIMAL,
            '10 point 5'
        );
    }

    // Integer number tests
    public function test_create_meta_field_number_integer_type(): void
    {
        $metaField = MetaFieldFacade::create(
            5,
            'App\Models\User',
            'number_integer_key',
            MetaFieldTypesEnum::NUMBER_INTEGER,
            9007199254740991
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_number_integer_value(): void
    {
        $this->expectException(MetaFieldException::class);
        MetaFieldFacade::create(
            5,
            'App\Models\User',
            'number_integer_key',
            MetaFieldTypesEnum::NUMBER_INTEGER,
            'invalid integer'
        );
    }

    // String type tests
    public function test_create_meta_field_single_line_text_field_type(): void
    {
        $metaField = MetaFieldFacade::create(
            3,
            'App\Models\User',
            'single_line_text_field_key',
            MetaFieldTypesEnum::SINGLE_LINE_TEXT_FIELD,
            fake()->jobTitle
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_single_line_text_field_value_is_not_more_than_255c(): void
    {
        $this->expectException(MetaFieldException::class);
        MetaFieldFacade::create(
            3,
            'App\Models\User',
            'single_line_text_field_key',
            MetaFieldTypesEnum::SINGLE_LINE_TEXT_FIELD,
            fake()->text(1000)
        );
    }

    // Longtext type tests
    public function test_create_meta_field_multi_line_text_field_type(): void
    {
        $metaField = MetaFieldFacade::create(
            3,
            'App\Models\User',
            'multi_line_text_field_key',
            MetaFieldTypesEnum::MULTI_LINE_TEXT_FIELD,
            fake()->text(1000)
        );
        $this->assertModelExists($metaField);
    }

    // Json type tests
    public function test_create_meta_field_json_type_object_value(): void
    {
        $metaField = MetaFieldFacade::create(
            5,
            'App\Models\User',
            'json_key',
            MetaFieldTypesEnum::JSON,
            json_encode(["ingredient" => "flour", "amount" => 0.3])
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_json_type_array_value(): void
    {
        $metaField = MetaFieldFacade::create(
            5,
            'App\Models\User',
            'json_key',
            MetaFieldTypesEnum::JSON,
            ["ingredient" => "flour", "amount" => 0.3]
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_json_type_string_value(): void
    {
        $metaField = MetaFieldFacade::create(
            5,
            'App\Models\User',
            'json_key',
            MetaFieldTypesEnum::JSON,
            "PHP is not dead"
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_json_type_number_value(): void
    {
        $metaField = MetaFieldFacade::create(
            5,
            'App\Models\User',
            'json_key',
            MetaFieldTypesEnum::JSON,
            1111110000.000112
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_json_type_boolean_value(): void
    {
        $metaField = MetaFieldFacade::create(
            5,
            'App\Models\User',
            'json_key',
            MetaFieldTypesEnum::JSON,
            false
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_json_type_null_value(): void
    {
        $metaField = MetaFieldFacade::create(
            5,
            'App\Models\User',
            'json_key',
            MetaFieldTypesEnum::JSON,
            null
        );
        $this->assertModelExists($metaField);
    }

    // Volume tests
    public function test_create_meta_field_volume_success_payload(): void
    {
        $metaField = MetaFieldFacade::create(
            5,
            'App\Models\User',
            'volume_key',
            MetaFieldTypesEnum::VOLUME,
            ['value' => 20.0, 'unit' => 'ml']
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_invalid_volume(): void
    {
        $this->expectException(MetaFieldException::class);
        MetaFieldFacade::create(
            5,
            'App\Models\User',
            'volume_key',
            MetaFieldTypesEnum::VOLUME,
            "Value should be an array"
        );
    }

    public function test_create_meta_field_invalid_unit_volume(): void
    {
        $this->expectException(MetaFieldException::class);
        MetaFieldFacade::create(
            5,
            'App\Models\User',
            'volume_key',
            MetaFieldTypesEnum::VOLUME,
            ['value' => 20.0, 'unit' => 'foo']
        );
    }

    public function test_create_meta_field_invalid_value_volume(): void
    {
        $this->expectException(MetaFieldException::class);
        MetaFieldFacade::create(
            5,
            'App\Models\User',
            'volume_key',
            MetaFieldTypesEnum::VOLUME,
            ['value' => 'twenty', 'unit' => MetaFieldVolumeUnitsEnum::ML]
        );
    }

    // Weight tests
    public function test_create_meta_field_valid_weight(): void
    {
        $metaField = MetaFieldFacade::create(
            5,
            'App\Models\User',
            'weight_key',
            MetaFieldTypesEnum::WEIGHT,
            ['value' => 2.5, 'unit' => MetaFieldWeightUnitsEnum::KG]
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_invalid_weight(): void
    {
        $this->expectException(MetaFieldException::class);
        MetaFieldFacade::create(
            5,
            'App\Models\User',
            'weight_key',
            MetaFieldTypesEnum::WEIGHT,
            "Value should be an array"
        );
    }

    public function test_create_meta_field_invalid_unit_weight(): void
    {
        $this->expectException(MetaFieldException::class);
        MetaFieldFacade::create(
            5,
            'App\Models\User',
            'weight_key',
            MetaFieldTypesEnum::WEIGHT,
            ['value' => 20.0, 'unit' => 'foo']
        );
    }

    public function test_create_meta_field_invalid_value_weight(): void
    {
        $this->expectException(MetaFieldException::class);
        MetaFieldFacade::create(
            5,
            'App\Models\User',
            'weight_key',
            MetaFieldTypesEnum::WEIGHT,
            ['value' => 'twenty', 'unit' => MetaFieldWeightUnitsEnum::KG]
        );
    }

    // Dimension tests
    public function test_create_meta_field_valid_dimension(): void
    {
        $metaField = MetaFieldFacade::create(
            5,
            'App\Models\User',
            'dimension_key',
            MetaFieldTypesEnum::DIMENSION,
            ['value' => 2.5, 'unit' => MetaFieldDimensionUnitsEnum::CM]
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_invalid_dimension(): void
    {
        $this->expectException(MetaFieldException::class);
        MetaFieldFacade::create(
            5,
            'App\Models\User',
            'dimension_key',
            MetaFieldTypesEnum::DIMENSION,
            "Value should be an array"
        );
    }

    public function test_create_meta_field_invalid_unit_dimension(): void
    {
        $this->expectException(MetaFieldException::class);
        MetaFieldFacade::create(
            5,
            'App\Models\User',
            'dimension_key',
            MetaFieldTypesEnum::DIMENSION,
            ['value' => 20.0, 'unit' => 'foo']
        );
    }

    public function test_create_meta_field_invalid_value_dimension(): void
    {
        $this->expectException(MetaFieldException::class);
        MetaFieldFacade::create(
            5,
            'App\Models\User',
            'dimension_key',
            MetaFieldTypesEnum::DIMENSION,
            ['value' => 'twenty', 'unit' => MetaFieldDimensionUnitsEnum::CM]
        );
    }

    /**
     * Meta field update test
     */
    public function test_update_meta_field_invalid_id(): void
    {
        $this->expectException(MetaFieldException::class);
        MetaFieldFacade::update(
            999999999999999,
            5,
            'App\Models\User',
            'weight_key',
            'MetaFieldTypesEnum::WEIGHT',
            ['value' => 20, 'unit' => MetaFieldWeightUnitsEnum::KG]
        );
    }

    /**
     * Meta field delete test
     */
    public function test_delete_meta_field_invalid_id(): void
    {
        $this->expectException(MetaFieldException::class);
        MetaFieldFacade::delete(999999999999999);
    }

    public function test_delete_meta_field_valid_id(): void
    {
        $metaField = MetaFieldFacade::create(
            5,
            'App\Models\User',
            'weight_key',
            MetaFieldTypesEnum::WEIGHT,
            ['value' => 2.5, 'unit' => MetaFieldWeightUnitsEnum::KG]
        );
        $deleteMetaField = MetaFieldFacade::delete($metaField->id);
        $this->assertTrue($deleteMetaField);
    }

    public function test_delete_meta_field_by_owner(): void
    {
        MetaFieldFacade::create(
            5,
            'App\Models\User',
            'weight_key',
            MetaFieldTypesEnum::WEIGHT,
            ['value' => 2.5, 'unit' => MetaFieldWeightUnitsEnum::KG]
        );
        $deleteMetaField = MetaFieldFacade::deleteByOwner(5, 'App\Models\User');
        $this->assertNull($deleteMetaField);
    }

    /**
     * Meta field get test
     */
    public function test_get_meta_field_by_owner(): void
    {
        MetaFieldFacade::create(
            5,
            'App\Models\User',
            'weight_key',
            MetaFieldTypesEnum::WEIGHT,
            ['value' => 2.5, 'unit' => MetaFieldWeightUnitsEnum::KG]
        );

        $metaFields = MetaFieldFacade::findByOwner(5, 'App\Models\User');
        $this->assertTrue($metaFields->count() > 0);
    }

    public function test_get_meta_field_by_unknown_owner(): void
    {
        $metaFields = MetaFieldFacade::findByOwner(5000000000000, 'App\Models\User');
        $this->assertTrue($metaFields->count() == 0);
    }

    public function test_get_meta_field_by_owner_key(): void
    {
        MetaFieldFacade::create(
            5,
            'App\Models\User',
            'weight_key',
            MetaFieldTypesEnum::WEIGHT,
            ['value' => 2.5, 'unit' => MetaFieldWeightUnitsEnum::KG]
        );

        $metaFields = MetaFieldFacade::findByKey(5, 'App\Models\User', 'weight_key');
        $this->assertTrue($metaFields->count() > 0);
    }

    public function test_get_meta_field_by_unknown_owner_key(): void
    {
        $metaFields = MetaFieldFacade::findByKey(5000000000000, 'App\Models\User', 'number_integer_key');
        $this->assertTrue($metaFields->count() == 0);
    }

    public function test_get_meta_field_by_owner_unknown_key(): void
    {
        $metaFields = MetaFieldFacade::findByKey(5, 'App\Models\User', 'fake_key');
        $this->assertTrue($metaFields->count() == 0);
    }

    public function test_get_meta_field_owner_by_key_value(): void
    {
        MetaFieldFacade::create(
            5,
            'App\Models\User',
            'weight_key',
            MetaFieldTypesEnum::WEIGHT,
            ['value' => 2.5, 'unit' => MetaFieldWeightUnitsEnum::KG]
        );

        $metaFields = MetaFieldFacade::findOwnerByValue('weight_key', MetaFieldTypesEnum::WEIGHT, '2.5');
        $this->assertTrue($metaFields->count() > 0);
    }
}
