<?php

namespace MoveOn\MetaField\Services;

use Carbon\Carbon;
use MoveOn\MetaField\Enum\MetaFieldDimensionUnitsEnum;
use MoveOn\MetaField\Enum\MetaFieldTypesEnum;
use MoveOn\MetaField\Enum\MetaFieldVolumeUnitsEnum;
use MoveOn\MetaField\Enum\MetaFieldWeightUnitsEnum;
use MoveOn\MetaField\Exception\MetaFieldException;
use MoveOn\MetaField\Repositories\MetaFieldRepository;

class MetaFieldService
{
    public function __construct(
        private MetaFieldRepository $metaFieldRepository
    )
    {
    }

    public function findByOwner(int $owner_id, string $owner_type)
    {
        return $this->metaFieldRepository->getAll($owner_id, $owner_type);
    }

    public function findByKey(int $owner_id, string $owner_type, string $key)
    {
        return $this->metaFieldRepository->getAll($owner_id, $owner_type, $key);
    }

    public function findByRegion(int $owner_id, string $owner_type, int $region_id)
    {
        return $this->metaFieldRepository->getAll($owner_id, $owner_type, null, $region_id);
    }

    public function findByRegionKey(int $owner_id, string $owner_type, int $region_id, string $key)
    {
        return $this->metaFieldRepository->getAll($owner_id, $owner_type, $key, $region_id);
    }

    public function findOwnerByValue(string $key, string $type, mixed $value)
    {
        return $this->metaFieldRepository->getAll(key: $key, type: $type, value: $value);
    }

    /**
     * Create meta field
     *
     * @param int $owner_id
     * @param string $owner_type
     * @param string $key
     * @param string $type
     * @param mixed $value
     * @param int|null $region_id
     * @return mixed
     * @throws MetaFieldException
     * @throws \Throwable
     */
    public function create(int    $owner_id,
        string $owner_type,
        string $key,
        string $type,
        mixed  $value,
        int    $region_id = null): mixed
    {
        try {
            $this->validateData($type, $value);

            $payload = [
                'owner_id'                              => $owner_id,
                'owner_type'                            => $owner_type,
                'key'                                   => $key,
                'type'                                  => $type,
                MetaFieldTypesEnum::VALUE_COLUMN[$type] => $type == MetaFieldTypesEnum::DATE_TIME ? Carbon::parse($value)->toIso8601String() : $value,
                'region_id'                             => $region_id
            ];

            $metaField = $this->metaFieldRepository->create($payload);
        } catch (\Exception $exception) {
            throw new MetaFieldException($exception->getMessage());
        }

        return $metaField;
    }

    /**
     * Create meta field
     *
     * @param  int  $owner_id
     * @param  string  $owner_type
     * @param  array  $kv_pair
     * @param  int|null  $region_id
     *
     * @return mixed
     * @throws MetaFieldException
     */
    public function createMultiple(
        int    $owner_id,
        string $owner_type,
        array  $kv_pair,
        int    $region_id = null): void
    {
        try {
            $payloadItems = [];
            foreach ($kv_pair as $kv) {
                $this->validateData($kv["type"], $kv["value"]);
                $payloadItems[] = [
                    'owner_id'                              => $owner_id,
                    'owner_type'                            => $owner_type,
                    'key'                                   => $kv['key'],
                    'type'                                  => $kv["type"],
                    MetaFieldTypesEnum::VALUE_COLUMN[$kv["type"]] => $kv["type"] == MetaFieldTypesEnum::DATE_TIME
                        ? Carbon::parse($kv["value"])->toIso8601String() : $kv["value"],
                    'region_id'                             => $region_id
                ];
            }

            $this->metaFieldRepository->createMultiple($payloadItems);
        } catch (\Exception $exception) {
            throw new MetaFieldException($exception->getMessage());
        }
    }

    /**
     * Update meta field
     *
     * @param int $id
     * @param int $owner_id
     * @param string $owner_type
     * @param string $key
     * @param string $type
     * @param mixed $value
     * @param int|null $region_id
     * @return mixed
     * @throws MetaFieldException
     * @throws \Throwable
     */
    public function update(
        int    $id,
        int    $owner_id,
        string $owner_type,
        string $key,
        string $type,
        mixed  $value,
        int    $region_id = null): mixed
    {
        try {
            $this->validateData($type, $value);

            $payload = [
                'owner_id'                              => $owner_id,
                'owner_type'                            => $owner_type,
                'key'                                   => $key,
                'type'                                  => $type,
                MetaFieldTypesEnum::VALUE_COLUMN[$type] => $type == MetaFieldTypesEnum::DATE_TIME ? Carbon::parse($value)->toIso8601String() : $value,
                'region_id'                             => $region_id
            ];

            $metaField = $this->metaFieldRepository->update($id, $payload);
        } catch (\Exception $exception) {
            throw new MetaFieldException($exception->getMessage());
        }

        return $metaField;
    }

    /**
     * Delete meta field
     *
     * @param int $id
     * @return mixed
     * @throws MetaFieldException
     */
    public function delete(int $id): mixed
    {
        try {
            $metaField = $this->metaFieldRepository->delete($id);
        } catch (\Exception $exception) {
            throw new MetaFieldException($exception->getMessage());
        }

        return $metaField;
    }

    /**
     * Update meta field
     *
     * @param int $id
     * @param int $owner_id
     * @param string $owner_type
     * @param string $key
     * @param string $type
     * @param mixed $value
     * @param int|null $region_id
     * @return mixed
     * @throws MetaFieldException
     */
    public function deleteByOwner(
        int    $owner_id,
        string $owner_type
    ): mixed
    {
        try {
            $metaField = $this->metaFieldRepository->deleteByOwner($id, $payload);
        } catch (\Exception $exception) {
            throw new MetaFieldException($exception->getMessage());
        }

        return $metaField;
    }

    /**
     * @param $type
     * @param $value
     * @return void
     * @throws MetaFieldException
     * @throws \Throwable
     */
    private function validateData($type, $value): void
    {
        throw_if(!isset(MetaFieldTypesEnum::VALUE_COLUMN[$type]), new MetaFieldException('Invalid meta field type provided.'));

        // boolean failure
        throw_if($type == MetaFieldTypesEnum::BOOLEAN && !is_bool($value), new MetaFieldException('Invalid value provided.'));

        // date_time failure
        if ($type == MetaFieldTypesEnum::DATE_TIME) {
            try {
                Carbon::parse($value);
            } catch (\Exception $e) {
                throw new MetaFieldException('Invalid date time provided.');
            }
        }

        // number_decimal failure
        if ($type == MetaFieldTypesEnum::NUMBER_DECIMAL && preg_match('/^-?\d+(\.\d+)?$/', $value) !== 1) {
            throw new MetaFieldException('Invalid decimal number provided.');
        }

        // number_integer failure
        if ($type == MetaFieldTypesEnum::NUMBER_INTEGER && filter_var($value, FILTER_VALIDATE_INT) === false) {
            throw new MetaFieldException('Invalid integer number provided.');
        }

        // single_line_text_field failure
        if ($type == MetaFieldTypesEnum::SINGLE_LINE_TEXT_FIELD && strlen($value) > 255) {
            throw new MetaFieldException('Invalid string provided.');
        }

        // volume failure
        if ($type == MetaFieldTypesEnum::VOLUME) {
            throw_if(!is_array($value), new MetaFieldException('Volume must be an array with value and unit keys.'));
            throw_if(!isset($value['value']) || !isset($value['unit']), new MetaFieldException('Invalid volume key provided.'));
            throw_if(!in_array($value['unit'], MetaFieldVolumeUnitsEnum::values()), new MetaFieldException('Invalid volume unit provided.'));
            throw_if(!is_numeric($value['value']), new MetaFieldException('Invalid volume value provided.'));
        }

        // weight failure
        if ($type == MetaFieldTypesEnum::WEIGHT) {
            throw_if(!is_array($value), new MetaFieldException('Weight must be an array with value and unit keys.'));
            throw_if(!isset($value['value']) || !isset($value['unit']), new MetaFieldException('Invalid weight key provided.'));
            throw_if(!in_array($value['unit'], MetaFieldWeightUnitsEnum::values()), new MetaFieldException('Invalid weight unit provided.'));
            throw_if(!is_numeric($value['value']), new MetaFieldException('Invalid weight value provided.'));
        }

        // dimension failure
        if ($type == MetaFieldTypesEnum::DIMENSION) {
            throw_if(!is_array($value), new MetaFieldException('Dimension must be an array with value and unit keys.'));
            throw_if(!isset($value['value']) || !isset($value['unit']), new MetaFieldException('Invalid dimension key provided.'));
            throw_if(!in_array($value['unit'], MetaFieldDimensionUnitsEnum::values()), new MetaFieldException('Invalid dimension unit provided.'));
            throw_if(!is_numeric($value['value']), new MetaFieldException('Invalid dimension value provided.'));
        }
    }
}
