<?php

namespace  MoveOn\MetaField\Tests\Unit;

use MoveOn\MetaField\Enum\MetaFieldDimensionUnitsEnum;
use MoveOn\MetaField\Enum\MetaFieldTypesEnum;
use MoveOn\MetaField\Enum\MetaFieldVolumeUnitsEnum;
use MoveOn\MetaField\Enum\MetaFieldWeightUnitsEnum;
use MoveOn\MetaField\Exception\MetaFieldException;
use MoveOn\MetaField\Models\MetaField;
use MoveOn\MetaField\Services\MetaFieldService;
use MoveOn\MetaField\Tests\TestCase;

class MetaFieldServiceTest extends TestCase
{
    public $metaFieldService;

    public function setUp(): void
    {
        parent::setUp();
        $this->metaFieldService = app(MetaFieldService::class);
    }

    public function test__construct()
    {
        MetaField::query()->truncate();
        $this->assertTrue(true);
    }

    /**
     * Meta field create test
     */

    // Boolean tests
    public function test_create_meta_field_boolean_type(): void
    {
        $metaField = $this->metaFieldService->create(
            4,
            'App\Models\User',
            'boolean_key',
            MetaFieldTypesEnum::BOOLEAN,
            true
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_boolean_type_value(): void
    {
        $this->expectException(MetaFieldException::class);
        $this->metaFieldService->create(
            4,
            'App\Models\User',
            'boolean_key',
            MetaFieldTypesEnum::BOOLEAN,
            "instead of boolean"
        );
    }

    // Date time tests
    public function test_create_meta_field_date_time_type(): void
    {
        $metaField = $this->metaFieldService->create(
            5,
            'App\Models\User',
            'date_time_key',
            MetaFieldTypesEnum::DATE_TIME,
            now()
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_date_time_value(): void
    {
        $this->expectException(MetaFieldException::class);
        $this->metaFieldService->create(
            5,
            'App\Models\User',
            'date_time_key',
            MetaFieldTypesEnum::DATE_TIME,
            'invalid date time value'
        );
    }

    // Decimal number tests
    public function test_create_meta_field_number_decimal_type(): void
    {
        $metaField = $this->metaFieldService->create(
            6,
            'App\Models\User',
            'number_decimal_key',
            MetaFieldTypesEnum::NUMBER_DECIMAL,
            543543354.544520
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_number_decimal_value(): void
    {
        $this->expectException(MetaFieldException::class);
        $this->metaFieldService->create(
            6,
            'App\Models\User',
            'number_decimal_key',
            MetaFieldTypesEnum::NUMBER_DECIMAL,
            '10 point 5'
        );
    }

    // Integer number tests
    public function test_create_meta_field_number_integer_type(): void
    {
        $metaField = $this->metaFieldService->create(
            5,
            'App\Models\User',
            'number_integer_key',
            MetaFieldTypesEnum::NUMBER_INTEGER,
            9007199254740991
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_number_integer_value(): void
    {
        $this->expectException(MetaFieldException::class);
        $this->metaFieldService->create(
            5,
            'App\Models\User',
            'number_integer_key',
            MetaFieldTypesEnum::NUMBER_INTEGER,
            'invalid integer'
        );
    }

    // String type tests
    public function test_create_meta_field_single_line_text_field_type(): void
    {
        $metaField = $this->metaFieldService->create(
            3,
            'App\Models\User',
            'single_line_text_field_key',
            MetaFieldTypesEnum::SINGLE_LINE_TEXT_FIELD,
            fake()->jobTitle
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_single_line_text_field_value_is_not_more_than_255c(): void
    {
        $this->expectException(MetaFieldException::class);
        $this->metaFieldService->create(
            3,
            'App\Models\User',
            'single_line_text_field_key',
            MetaFieldTypesEnum::SINGLE_LINE_TEXT_FIELD,
            fake()->text(1000)
        );
    }

    // Longtext type tests
    public function test_create_meta_field_multi_line_text_field_type(): void
    {
        $metaField = $this->metaFieldService->create(
            3,
            'App\Models\User',
            'multi_line_text_field_key',
            MetaFieldTypesEnum::MULTI_LINE_TEXT_FIELD,
            fake()->text(1000)
        );
        $this->assertModelExists($metaField);
    }

    // Json type tests
    public function test_create_meta_field_json_type_object_value(): void
    {
        $metaField = $this->metaFieldService->create(
            5,
            'App\Models\User',
            'json_key',
            MetaFieldTypesEnum::JSON,
            json_encode(["ingredient" => "flour", "amount" => 0.3])
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_json_type_array_value(): void
    {
        $metaField = $this->metaFieldService->create(
            5,
            'App\Models\User',
            'json_key',
            MetaFieldTypesEnum::JSON,
            ["ingredient" => "flour", "amount" => 0.3]
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_json_type_string_value(): void
    {
        $metaField = $this->metaFieldService->create(
            5,
            'App\Models\User',
            'json_key',
            MetaFieldTypesEnum::JSON,
            "PHP is not dead"
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_json_type_number_value(): void
    {
        $metaField = $this->metaFieldService->create(
            5,
            'App\Models\User',
            'json_key',
            MetaFieldTypesEnum::JSON,
            1111110000.000112
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_json_type_boolean_value(): void
    {
        $metaField = $this->metaFieldService->create(
            5,
            'App\Models\User',
            'json_key',
            MetaFieldTypesEnum::JSON,
            false
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_json_type_null_value(): void
    {
        $metaField = $this->metaFieldService->create(
            5,
            'App\Models\User',
            'json_key',
            MetaFieldTypesEnum::JSON,
            null
        );
        $this->assertModelExists($metaField);
    }

    // Volume tests
    public function test_create_meta_field_volume_success_payload(): void
    {
        $metaField = $this->metaFieldService->create(
            5,
            'App\Models\User',
            'volume_key',
            MetaFieldTypesEnum::VOLUME,
            ['value' => 20.0, 'unit' => 'ml']
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_invalid_volume(): void
    {
        $this->expectException(MetaFieldException::class);
        $this->metaFieldService->create(
            5,
            'App\Models\User',
            'volume_key',
            MetaFieldTypesEnum::VOLUME,
            "Value should be an array"
        );
    }

    public function test_create_meta_field_invalid_unit_volume(): void
    {
        $this->expectException(MetaFieldException::class);
        $this->metaFieldService->create(
            5,
            'App\Models\User',
            'volume_key',
            MetaFieldTypesEnum::VOLUME,
            ['value' => 20.0, 'unit' => 'foo']
        );
    }

    public function test_create_meta_field_invalid_value_volume(): void
    {
        $this->expectException(MetaFieldException::class);
        $this->metaFieldService->create(
            5,
            'App\Models\User',
            'volume_key',
            MetaFieldTypesEnum::VOLUME,
            ['value' => 'twenty', 'unit' => MetaFieldVolumeUnitsEnum::ML]
        );
    }

    // Weight tests
    public function test_create_meta_field_valid_weight(): void
    {
        $metaField = $this->metaFieldService->create(
            5,
            'App\Models\User',
            'weight_key',
            MetaFieldTypesEnum::WEIGHT,
            ['value' => 2.5, 'unit' => MetaFieldWeightUnitsEnum::KG]
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_invalid_weight(): void
    {
        $this->expectException(MetaFieldException::class);
        $this->metaFieldService->create(
            5,
            'App\Models\User',
            'weight_key',
            MetaFieldTypesEnum::WEIGHT,
            "Value should be an array"
        );
    }

    public function test_create_meta_field_invalid_unit_weight(): void
    {
        $this->expectException(MetaFieldException::class);
        $this->metaFieldService->create(
            5,
            'App\Models\User',
            'weight_key',
            MetaFieldTypesEnum::WEIGHT,
            ['value' => 20.0, 'unit' => 'foo']
        );
    }

    public function test_create_meta_field_invalid_value_weight(): void
    {
        $this->expectException(MetaFieldException::class);
        $this->metaFieldService->create(
            5,
            'App\Models\User',
            'weight_key',
            MetaFieldTypesEnum::WEIGHT,
            ['value' => 'twenty', 'unit' => MetaFieldWeightUnitsEnum::KG]
        );
    }

    // Dimension tests
    public function test_create_meta_field_valid_dimension(): void
    {
        $metaField = $this->metaFieldService->create(
            5,
            'App\Models\User',
            'dimension_key',
            MetaFieldTypesEnum::DIMENSION,
            ['value' => 2.5, 'unit' => MetaFieldDimensionUnitsEnum::CM]
        );
        $this->assertModelExists($metaField);
    }

    public function test_create_meta_field_invalid_dimension(): void
    {
        $this->expectException(MetaFieldException::class);
        $this->metaFieldService->create(
            5,
            'App\Models\User',
            'dimension_key',
            MetaFieldTypesEnum::DIMENSION,
            "Value should be an array"
        );
    }

    public function test_create_meta_field_invalid_unit_dimension(): void
    {
        $this->expectException(MetaFieldException::class);
        $this->metaFieldService->create(
            5,
            'App\Models\User',
            'dimension_key',
            MetaFieldTypesEnum::DIMENSION,
            ['value' => 20.0, 'unit' => 'foo']
        );
    }

    public function test_create_meta_field_invalid_value_dimension(): void
    {
        $this->expectException(MetaFieldException::class);
        $this->metaFieldService->create(
            5,
            'App\Models\User',
            'dimension_key',
            MetaFieldTypesEnum::DIMENSION,
            ['value' => 'twenty', 'unit' => MetaFieldDimensionUnitsEnum::CM]
        );
    }

    /**
     * Meta field update test
     */
    public function test_update_meta_field_invalid_id(): void
    {
        $this->expectException(MetaFieldException::class);
        $this->metaFieldService->update(
            999999999999999,
            5,
            'App\Models\User',
            'weight_key',
            'MetaFieldTypesEnum::WEIGHT',
            ['value' => 20, 'unit' => MetaFieldWeightUnitsEnum::KG]
        );
    }

    /**
     * Meta field delete test
     */
    public function test_delete_meta_field_invalid_id(): void
    {
        $this->expectException(MetaFieldException::class);
        $this->metaFieldService->delete(999999999999999);
    }

    public function test_delete_meta_field_valid_id(): void
    {
        $metaField = $this->metaFieldService->create(
            5,
            'App\Models\User',
            'weight_key',
            MetaFieldTypesEnum::WEIGHT,
            ['value' => 2.5, 'unit' => MetaFieldWeightUnitsEnum::KG]
        );
        $deleteMetaField = $this->metaFieldService->delete($metaField->id);
        $this->assertTrue($deleteMetaField);
    }

    /**
     * Meta field get test
     */
    public function test_get_meta_field_by_owner(): void
    {
        $this->metaFieldService->create(
            5,
            'App\Models\User',
            'weight_key',
            MetaFieldTypesEnum::WEIGHT,
            ['value' => 2.5, 'unit' => MetaFieldWeightUnitsEnum::KG]
        );

        $metaFields = $this->metaFieldService->findByOwner(5, 'App\Models\User');
        $this->assertTrue($metaFields->count() > 0);
    }

    public function test_get_meta_field_by_unknown_owner(): void
    {
        $metaFields = $this->metaFieldService->findByOwner(5000000000000, 'App\Models\User');
        $this->assertTrue($metaFields->count() == 0);
    }

    public function test_get_meta_field_by_owner_key(): void
    {
        $this->metaFieldService->create(
            5,
            'App\Models\User',
            'weight_key',
            MetaFieldTypesEnum::WEIGHT,
            ['value' => 2.5, 'unit' => MetaFieldWeightUnitsEnum::KG]
        );

        $metaFields = $this->metaFieldService->findByKey(5, 'App\Models\User', 'weight_key');
        $this->assertTrue($metaFields->count() > 0);
    }

    public function test_get_meta_field_by_unknown_owner_key(): void
    {
        $metaFields = $this->metaFieldService->findByKey(5000000000000, 'App\Models\User', 'number_integer_key');
        $this->assertTrue($metaFields->count() == 0);
    }

    public function test_get_meta_field_by_owner_unknown_key(): void
    {
        $metaFields = $this->metaFieldService->findByKey(5, 'App\Models\User', 'fake_key');
        $this->assertTrue($metaFields->count() == 0);
    }

    public function test_get_meta_field_owner_by_key_value(): void
    {
        $this->metaFieldService->create(
            5,
            'App\Models\User',
            'weight_key',
            MetaFieldTypesEnum::WEIGHT,
            ['value' => 2.5, 'unit' => MetaFieldWeightUnitsEnum::KG]
        );

        $metaFields = $this->metaFieldService->findOwnerByValue('weight_key', MetaFieldTypesEnum::WEIGHT, '2.5');
        $this->assertTrue($metaFields->count() > 0);
    }
}
