<?php

namespace MoveOn\MetaField\Services;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Collection;
use MoveOn\MetaField\Enum\MetaFieldDimensionUnitsEnum;
use MoveOn\MetaField\Enum\MetaFieldTypesEnum;
use MoveOn\MetaField\Enum\MetaFieldVolumeUnitsEnum;
use MoveOn\MetaField\Enum\MetaFieldWeightUnitsEnum;
use MoveOn\MetaField\Exception\MetaFieldException;
use MoveOn\MetaField\Repositories\MetaFieldRepository;

class MetaFieldService
{
    /**
     * @param  int  $ownerId
     * @param  string  $ownerType
     *
     * @return Collection
     */
    public static function findByOwner(int $ownerId, string $ownerType) 
    {
        return MetaFieldRepository::make()->getAll($ownerId, $ownerType);
    }

    /**
     * @param  int  $ownerId
     * @param  string  $ownerType
     * @param  string  $key
     *
     * @return Collection
     */
    public function findByKey(int $ownerId, string $ownerType, string $key)
    {
        return MetaFieldRepository::make()->getAll($ownerId, $ownerType, $key);
    }

    /**
     * @param  int  $ownerId
     * @param  string  $ownerType
     * @param  int  $regionId
     *
     * @return Collection
     */
    public function findByRegion(int $ownerId, string $ownerType, int $regionId)
    {
        return MetaFieldRepository::make()->getAll($ownerId, $ownerType, null, $regionId);
    }

    /**
     * @param  int  $ownerId
     * @param  string  $ownerType
     * @param  int  $regionId
     * @param  string  $key
     *
     * @return Collection
     */
    public function findByRegionKey(int $ownerId, string $ownerType, int $regionId, string $key)
    {
        return MetaFieldRepository::make()->getAll($ownerId, $ownerType, $key, $regionId);
    }

    /**
     * @param  string  $key
     * @param  string  $type
     * @param  mixed  $value
     *
     * @return Collection
     */
    public function findOwnerByValue(string $key, string $type, mixed $value)
    {
        return MetaFieldRepository::make()->getAll(key: $key, type: $type, value: $value);
    }

    /**
     * Create meta field
     *
     * @param int $ownerId
     * @param string $ownerType
     * @param string $key
     * @param string $type
     * @param mixed $value
     * @param int|null $regionId
     * @return mixed
     * @throws MetaFieldException
     * @throws \Throwable
     */
    public function create(
        int    $ownerId,
        string $ownerType,
        string $key,
        string $type,
        mixed  $value,
        int    $regionId = null): mixed
    {
        try {
            $this->validateData($type, $value);

            $payload = [
                'owner_id'                              => $ownerId,
                'owner_type'                            => $ownerType,
                'key'                                   => $key,
                'type'                                  => $type,
                MetaFieldTypesEnum::VALUE_COLUMN[$type] => $type == MetaFieldTypesEnum::DATE_TIME ? Carbon::parse($value)->toIso8601String() : $value,
                'region_id'                             => $regionId
            ];

            $metaField = MetaFieldRepository::make()->create($payload);
        } catch (\Exception $exception) {
            throw new MetaFieldException($exception->getMessage());
        }

        return $metaField;
    }

    /**
     * Create meta field
     *
     * @param  int  $ownerId
     * @param  string  $ownerType
     * @param  array  $kvPair
     * @param  int|null  $regionId
     *
     * @return mixed
     * @throws MetaFieldException|\Throwable
     */
    public function createMultiple(
        int    $ownerId,
        string $ownerType,
        array  $kvPair,
        int    $regionId = null): void
    {
        try {
            $payloadItems = [];
            foreach ($kvPair as $kv) {
                $this->validateData($kv["type"], $kv["value"]);
                $payloadItems[] = [
                    'owner_id'                              => $ownerId,
                    'owner_type'                            => $ownerType,
                    'key'                                   => $kv['key'],
                    'type'                                  => $kv["type"],
                    MetaFieldTypesEnum::VALUE_COLUMN[$kv["type"]] => $kv["type"] == MetaFieldTypesEnum::DATE_TIME
                        ? Carbon::parse($kv["value"])->toIso8601String() : $kv["value"],
                    'region_id'                             => $regionId
                ];
            }

            MetaFieldRepository::make()->createMultiple($payloadItems);
        } catch (\Exception $exception) {
            throw new MetaFieldException($exception->getMessage());
        }
    }

    /**
     * Update meta field
     *
     * @param int $id
     * @param int $ownerId
     * @param string $ownerType
     * @param string $key
     * @param string $type
     * @param mixed $value
     * @param int|null $regionId
     * @return mixed
     * @throws MetaFieldException
     * @throws \Throwable
     */
    public function update(
        int    $id,
        int    $ownerId,
        string $ownerType,
        string $key,
        string $type,
        mixed  $value,
        int    $regionId = null): mixed
    {
        try {
            $this->validateData($type, $value);

            $payload = [
                'owner_id'                              => $ownerId,
                'owner_type'                            => $ownerType,
                'key'                                   => $key,
                'type'                                  => $type,
                MetaFieldTypesEnum::VALUE_COLUMN[$type] => $type == MetaFieldTypesEnum::DATE_TIME ? Carbon::parse($value)->toIso8601String() : $value,
                'region_id'                             => $regionId
            ];

            $metaField = MetaFieldRepository::make()->update($id, $payload);
        } catch (\Exception $exception) {
            throw new MetaFieldException($exception->getMessage());
        }

        return $metaField;
    }

    /**
     * Delete meta field
     *
     * @param int $id
     * @return mixed
     * @throws MetaFieldException
     */
    public function delete(int $id): mixed
    {
        try {
            $metaField = MetaFieldRepository::make()->delete($id);
        } catch (\Exception $exception) {
            throw new MetaFieldException($exception->getMessage());
        }

        return $metaField;
    }

    /**
     * Update meta field
     *
     * @param  int  $ownerId
     * @param  string  $ownerType
     *
     * @throws MetaFieldException
     */
    public function deleteByOwner(
        int    $ownerId,
        string $ownerType
    ): void
    {
        try {
            MetaFieldRepository::make()->deleteByOwner($ownerId, $ownerType);
        } catch (\Exception $exception) {
            throw new MetaFieldException($exception->getMessage());
        }
    }

    /**
     * @param $type
     * @param $value
     * @return void
     * @throws MetaFieldException
     * @throws \Throwable
     */
    private function validateData($type, $value): void
    {
        throw_if(!isset(MetaFieldTypesEnum::VALUE_COLUMN[$type]), new MetaFieldException('Invalid meta field type provided.'));

        // boolean failure
        throw_if($type == MetaFieldTypesEnum::BOOLEAN && !is_bool($value), new MetaFieldException('Invalid value provided.'));

        // date_time failure
        if ($type == MetaFieldTypesEnum::DATE_TIME) {
            try {
                Carbon::parse($value);
            } catch (\Exception $e) {
                throw new MetaFieldException('Invalid date time provided.');
            }
        }

        // number_decimal failure
        if ($type == MetaFieldTypesEnum::NUMBER_DECIMAL && preg_match('/^-?\d+(\.\d+)?$/', $value) !== 1) {
            throw new MetaFieldException('Invalid decimal number provided.');
        }

        // number_integer failure
        if ($type == MetaFieldTypesEnum::NUMBER_INTEGER && filter_var($value, FILTER_VALIDATE_INT) === false) {
            throw new MetaFieldException('Invalid integer number provided.');
        }

        // single_line_text_field failure
        if ($type == MetaFieldTypesEnum::SINGLE_LINE_TEXT_FIELD && strlen($value) > 255) {
            throw new MetaFieldException('Invalid string provided.');
        }

        // volume failure
        if ($type == MetaFieldTypesEnum::VOLUME) {
            throw_if(!is_array($value), new MetaFieldException('Volume must be an array with value and unit keys.'));
            throw_if(!isset($value['value']) || !isset($value['unit']), new MetaFieldException('Invalid volume key provided.'));
            throw_if(!in_array($value['unit'], MetaFieldVolumeUnitsEnum::values()), new MetaFieldException('Invalid volume unit provided.'));
            throw_if(!is_numeric($value['value']), new MetaFieldException('Invalid volume value provided.'));
        }

        // weight failure
        if ($type == MetaFieldTypesEnum::WEIGHT) {
            throw_if(!is_array($value), new MetaFieldException('Weight must be an array with value and unit keys.'));
            throw_if(!isset($value['value']) || !isset($value['unit']), new MetaFieldException('Invalid weight key provided.'));
            throw_if(!in_array($value['unit'], MetaFieldWeightUnitsEnum::values()), new MetaFieldException('Invalid weight unit provided.'));
            throw_if(!is_numeric($value['value']), new MetaFieldException('Invalid weight value provided.'));
        }

        // dimension failure
        if ($type == MetaFieldTypesEnum::DIMENSION) {
            throw_if(!is_array($value), new MetaFieldException('Dimension must be an array with value and unit keys.'));
            throw_if(!isset($value['value']) || !isset($value['unit']), new MetaFieldException('Invalid dimension key provided.'));
            throw_if(!in_array($value['unit'], MetaFieldDimensionUnitsEnum::values()), new MetaFieldException('Invalid dimension unit provided.'));
            throw_if(!is_numeric($value['value']), new MetaFieldException('Invalid dimension value provided.'));
        }
    }
}
