<?php

namespace MoveOn\MetaField\Repositories;

use Exception;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use MoveOn\MetaField\Enum\MetaFieldTypesEnum;
use MoveOn\MetaField\Models\MetaField;
use MoveOn\MetaField\Traits\Makeable;

class MetaFieldRepository
{
    use Makeable;

    const MAX_RETRY = 5;

    /**
     * @param  int|null  $owner_id
     * @param  string|null  $owner_type
     * @param  string|null  $key
     * @param  int|null  $region_id
     * @param  string|null  $type
     * @param  mixed|null  $value
     *
     * @return \Illuminate\Database\Eloquent\Collection
     */
    public function getAll(int $owner_id = null, string $owner_type = null, string $key = null, int $region_id = null, string $type = null, mixed $value = null)
    {
        return MetaField::query()
                        ->when($owner_id, function ($q) use ($owner_id) {
                            return $q->where('owner_id', $owner_id);
                        })
                        ->when($owner_type, function ($q) use ($owner_type) {
                            return $q->where('owner_type', $owner_type);
                        })
                        ->when($key, function ($q) use ($key) {
                            return $q->where('key', $key);
                        })
                        ->when($region_id, function ($q) use ($region_id) {
                            return $q->where('region_id', $region_id);
                        })
                        ->when($type, function ($q) use ($type) {
                            return $q->where('type', $type);
                        })
                        ->when($type && $value, function ($q) use ($type, $value) {
                            return $q->where(MetaFieldTypesEnum::VALUE_COLUMN[$type], 'like', "%{$value}%");
                        })
                        ->get();
    }

    /**
     * @param array $payload
     * @return mixed
     * @throws Exception
     */
    public function create(array $payload)
    {
        try {
            DB::beginTransaction();
            $payload = array_merge($payload, [
                "etag"         => (string)Str::uuid(),
                "lock_version" => 1
            ]);
            $metaField = MetaField::create($payload);
            DB::commit();
            return $metaField;
        } catch (Exception $exception) {
            DB::rollBack();
            throw $exception;
        }
    }


    /**
     * Create multiple MetaFields for owner
     *
     * @param  array  $payloadItems
     *
     * @return void
     * @throws Exception
     */
    public function createMultiple(array $payloadItems)
    {
        try {
            DB::beginTransaction();
            $processedPayload = [];
            foreach ($payloadItems as $item) {
                $processedPayload[] = array_merge($item, [
                    "etag"         => (string)Str::uuid(),
                    "lock_version" => 1,
                    "created_at"   => now(),
                    "updated_at"   => now(),
                ]);
            }
            MetaField::insert($processedPayload);
            DB::commit();
        } catch (Exception $exception) {
            DB::rollBack();
            throw $exception;
        }
    }

    /**
     * @param  int  $id
     * @param  array  $changes
     *
     * @return mixed
     * @throws Exception
     */
    public function update(int $id, array $changes)
    {
        $attempt = 1;

        $metaField = MetaField::find($id);

        if (!$metaField) {
            throw new Exception('No data found for the given id!');
        }

        do {
            $changes = array_merge($changes, [
                "etag"         => (string) Str::uuid(),
                "lock_version" => $metaField->lock_version + 1
            ]);

            $updated = MetaField::where([
                ["id", $id],
                ["etag", $metaField->etag]
            ])->update($changes);

            $attempt++;
        } while (!$updated && $attempt <= self::MAX_RETRY);

        if (!$updated && $attempt > self::MAX_RETRY) {
            throw new Exception("Max retry exceeded during Meta Field update");
        }

        return $metaField->refresh();
    }

    /**
     * Delete the MetaField
     *
     * @param  int  $id
     *
     * @return mixed
     * @throws Exception
     */
    public function delete(int $id)
    {
        $metaField = MetaField::find($id);

        if (!$metaField) {
            throw new Exception('No data found for the given id!', 404);
        }

        return $metaField->delete();
    }

    /**
     * Delete MetaFields by owner
     *
     * @param  int  $ownerId
     * @param  string  $ownerKey
     *
     * @return void
     * @throws Exception
     */
    public function deleteByOwner(int $ownerId, string $ownerKey)
    {
        try {
            DB::beginTransaction();
            MetaField::where([
                [
                    "owner_id", "=", $ownerId
                ],
                [
                    "owner_type", "=", $ownerKey
                ],
            ])->delete();
            DB::commit();
        } catch (Exception $exception) {
            DB::rollBack();
            throw $exception;
        }
    }
}
