<?php

declare(strict_types=1);

namespace MoveOn\Ory\Guards;

use MoveOn\Ory\Entities\{CredentialEntityContract, InstanceEntity, InstanceEntityContract};
use MoveOn\Ory\Events;
use Ory\Client\Api\FrontendApi;
use Ory\Client\ApiException;
use MoveOn\Ory\Events\{TokenVerificationAttempting, TokenVerificationFailed, TokenVerificationSucceeded};
use MoveOn\Ory\Exceptions\{AuthenticationException, GuardException, GuardExceptionContract};
use Exception;
use Illuminate\Contracts\Auth\{Authenticatable, Guard, UserProvider};
use Illuminate\Contracts\Session\Session;
use Illuminate\Contracts\Support\{Arrayable, Jsonable};
use JsonSerializable;

use function in_array;
use function is_array;
use function is_int;
use function is_string;

/**
 * @internal
 *
 * @api
 */
abstract class GuardAbstract implements Guard
{
    protected ?CredentialEntityContract $credential = null;

    protected ?CredentialEntityContract $impersonating = null;

    protected ?int $impersonationSource = null;

    protected ?UserProvider $provider = null;

    protected ?Session $session = null;

    public function __construct(
        public string $name = '',
        protected ?array $config = null,
        protected ?InstanceEntityContract $sdk = null,
    ) {
    }

    final public function authenticate(): Authenticatable
    {
        if (($user = $this->user()) instanceof Authenticatable) {
            return $user;
        }

        throw new AuthenticationException(AuthenticationException::UNAUTHENTICATED);
    }

    final public function check(): bool
    {
        return $this->hasUser();
    }

    final public function getImposter(): ?CredentialEntityContract
    {
        return $this->impersonating;
    }

    final public function getImposterSource(): ?int
    {
        return $this->impersonationSource;
    }

    final public function getName(): string
    {
        return $this->name;
    }

    final public function getProvider(): UserProvider
    {
        if ($this->provider instanceof UserProvider) {
            return $this->provider;
        }

        $providerName = $this->config['provider'] ?? '';

        if (! is_string($providerName) || '' === $providerName) {
            // @codeCoverageIgnoreStart
            throw new GuardException(GuardExceptionContract::USER_PROVIDER_UNCONFIGURED);
            // @codeCoverageIgnoreEnd
        }

        $providerName = trim($providerName);
        $provider = app('auth')->createUserProvider($providerName);

        if ($provider instanceof UserProvider) {
            $this->provider = $provider;

            return $provider;
        }

        // @codeCoverageIgnoreStart
        throw new GuardException(sprintf(GuardExceptionContract::USER_PROVIDER_UNAVAILABLE, $providerName));
        // @codeCoverageIgnoreEnd
    }


    final public function getSession(): Session
    {

    }

    final public function guest(): bool
    {
        return ! $this->check();
    }

    final public function hasUser(): bool
    {
        return $this->user() instanceof Authenticatable;
    }

    final public function id(): string | null
    {
        $user = $this->user()?->getAuthIdentifier();

        if (is_string($user) || is_int($user)) {
            return (string) $user;
        }

        return null;
    }

    final public function isImpersonating(): bool
    {
        return $this->impersonating instanceof CredentialEntityContract;
    }

    final public function processToken(
        string $xSessionToken,
        array $cookie,
    ): ?array {
        Events::dispatch($event = new TokenVerificationAttempting(
            $xSessionToken,
            $cookie
        ));
        try {
            $decoded = $this->sdk()["main"]->toSession(
                xSessionToken: $xSessionToken,
                cookie: count($cookie) > 0 ? $cookie : null
            );
            $decoded = json_decode(json_encode(
                (array)$decoded->jsonSerialize()),
                true
            );
        } catch (ApiException $invalidTokenException) {
            try {
                $decoded = $this->sdk()["proxy"]->toSession(
                    xSessionToken: $xSessionToken,
                    cookie: count($cookie) > 0 ? $cookie : null
                );
                $decoded = json_decode(json_encode((array)$decoded->jsonSerialize()), true);
            } catch (ApiException $invalidTokenException) {
                Events::dispatch($event = new TokenVerificationFailed(
                    $xSessionToken,
                    $cookie,
                    $invalidTokenException));
                if ($event->throwException) {
                    // @codeCoverageIgnoreStart
                    throw $invalidTokenException;
                    // @codeCoverageIgnoreEnd
                }
                return null;
            }
        }
        Events::dispatch(new TokenVerificationSucceeded($xSessionToken, $cookie, $decoded));
        return $decoded;
    }

    final public function sdk(
        bool $reset = false,
    ): array {
        if (! $this->sdk instanceof InstanceEntityContract || $reset) {
            $configurationName = $this->config['configuration'] ?? $this->name;
            $this->sdk = InstanceEntity::create(
                guardConfigurationName: $configurationName,
            );
        }

        return [
            "main" => $this->sdk->getMainInstance(),
            "proxy" => $this->sdk->getProxyInstance()
        ];
    }

    /**
     * @codeCoverageIgnore
     */
    final public function service(): ?InstanceEntityContract
    {
        return $this->sdk;
    }

    final public function stopImpersonating(): void
    {
        $this->impersonating = null;
        $this->impersonationSource = null;
    }

    /**
     * @phpcsSuppress SlevomatCodingStandard.Functions.UnusedParameter
     *
     * @param array $credentials
     */
    final public function validate(
        array $credentials = [],
    ): bool {
        return false;
    }

    final public function viaRemember(): bool
    {
        return false;
    }

    abstract public function find(): ?CredentialEntityContract;

    abstract public function getCredential(): ?CredentialEntityContract;

    abstract public function setCredential(?CredentialEntityContract $credential = null): GuardContract;

    /**
     * Toggle the Guard's impersonation state. This should only be used by the Impersonate trait, and is not intended for use by end-users. It is public to allow for testing.
     *
     * @param CredentialEntityContract $credential
     */
    abstract public function setImpersonating(
        CredentialEntityContract $credential,
    ): self;

    abstract public function setUser(
        Authenticatable $user,
    ): void;

    abstract public function user(): ?Authenticatable;
}
