<?php

declare(strict_types=1);

namespace MoveOn\Ory;

use MoveOn\Ory\Events\{TokenVerificationAttempting, TokenVerificationFailed, TokenVerificationSucceeded};
use MoveOn\Ory\Guards\{AuthorizationGuardContract, GuardContract};
use MoveOn\Ory\{UserRepository, UserRepositoryContract};
use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Support\Facades\Cache;

use function is_string;

/**
 *
 * @api
 */
abstract class UserProviderAbstract
{
    protected ?UserRepositoryContract $repository = null;

    protected string $repositoryName = '';

    /**
     * The Eloquent user model.
     *
     * @var string
     */
    protected $model;

    public function __construct(
        public array $config = [],
    ) {
        $this->model    = $this->config["model"];
    }

    final public function getRepository(): UserRepositoryContract
    {
        return $this->repository ?? $this->resolveRepository();
    }

    /**
     * @phpcsSuppress SlevomatCodingStandard.Functions.UnusedParameter
     *
     * @param array $credentials
     */
    final public function retrieveByCredentials(array $credentials): ?Authenticatable{}

    /**
     * @phpcsSuppress SlevomatCodingStandard.Functions.UnusedParameter
     *
     * @codeCoverageIgnore
     *
     * @param mixed $identifier
     */
    final public function retrieveById($identifier): ?Authenticatable
    {
        return null;
    }

    /**
     * @psalm-suppress DocblockTypeContradiction
     *
     * @phpcsSuppress SlevomatCodingStandard.Functions.UnusedParameter
     *
     * @param mixed $identifier
     * @param mixed $token
     */
    final public function retrieveByToken($identifier, $token): ?Authenticatable
    {}

    final public function setRepository(string $repository): void
    {
        $this->resolveRepository($repository);
    }

    /**
     * @phpcsSuppress SlevomatCodingStandard.Functions.UnusedParameter
     *
     * @codeCoverageIgnore
     *
     * @param Authenticatable $user
     * @param mixed           $token
     */
    final public function updateRememberToken(Authenticatable $user, $token): void
    {}

    /**
     * @phpcsSuppress SlevomatCodingStandard.Functions.UnusedParameter
     *
     * @param Authenticatable $user
     * @param array           $credentials
     */
    final public function validateCredentials(
        Authenticatable $user,
        array $credentials,
    ): bool {
        return false;
    }

    protected function getConfiguration(
        string $key,
    ): array | string | null {
        return $this->config[$key] ?? null;
    }

    protected function getRepositoryName(): string
    {
        return $this->repositoryName;
    }

    protected function resolveRepository(
        ?string $repositoryName = null,
    ): UserRepositoryContract {
        $repository = $repositoryName;
        $repository ??= $this->getConfiguration('repository');
        $repository ??= UserRepository::class;

        if ($repository === $this->getRepositoryName()) {
            return $this->getRepository();
        }

        if (! is_string($repository)) {
            throw new BindingResolutionException('The configured Repository could not be loaded.');
        }

        if (! app()->bound($repository)) {
            try {
                app()->make($repository);
            } catch (BindingResolutionException) {
                throw new BindingResolutionException(sprintf('The configured Repository %s could not be loaded.', $model));
            }
        }

        $this->setRepositoryName($repository);

        return $this->repository = app($repository);
    }

    protected function setConfiguration(
        string $key,
        string $value,
    ): void {
        $this->config[$key] = $value;
    }

    protected function setRepositoryName(string $repositoryName): void
    {
        $this->setConfiguration('repository', $repositoryName);
        $this->repositoryName = $repositoryName;
    }
}
