<?php

declare(strict_types=1);

namespace MoveOn\Ory\Guards;

use MoveOn\Ory\Entities\{CredentialEntityContract, InstanceEntity, InstanceEntityContract};
use MoveOn\Ory\Events;
use Ory\Client\Api\FrontendApi;
use MoveOn\Ory\Events\{TokenVerificationAttempting, TokenVerificationFailed, TokenVerificationSucceeded};
use MoveOn\Ory\Exceptions\{AuthenticationException, GuardException, GuardExceptionContract};
use Exception;
use Illuminate\Contracts\Auth\{Authenticatable, Guard, UserProvider};
use Illuminate\Contracts\Session\Session;
use Illuminate\Contracts\Support\{Arrayable, Jsonable};
use JsonSerializable;

use function in_array;
use function is_array;
use function is_int;
use function is_string;

/**
 * @internal
 *
 * @api
 */
abstract class GuardAbstract implements Guard
{
    protected ?CredentialEntityContract $credential = null;

    protected ?CredentialEntityContract $impersonating = null;

    protected ?int $impersonationSource = null;

    protected ?UserProvider $provider = null;

    protected ?Session $session = null;

    public function __construct(
        public string $name = '',
        protected ?array $config = null,
        protected ?InstanceEntityContract $sdk = null,
    ) {
    }

    final public function authenticate(): Authenticatable
    {
        if (($user = $this->user()) instanceof Authenticatable) {
            return $user;
        }

        throw new AuthenticationException(AuthenticationException::UNAUTHENTICATED);
    }

    final public function check(): bool
    {
        return $this->hasUser();
    }

    final public function getImposter(): ?CredentialEntityContract
    {
        return $this->impersonating;
    }

    final public function getImposterSource(): ?int
    {
        return $this->impersonationSource;
    }

    final public function getName(): string
    {
        return $this->name;
    }

    final public function getProvider(): UserProvider
    {
        if ($this->provider instanceof UserProvider) {
            return $this->provider;
        }

        $providerName = $this->config['provider'] ?? '';

        if (! is_string($providerName) || '' === $providerName) {
            // @codeCoverageIgnoreStart
            throw new GuardException(GuardExceptionContract::USER_PROVIDER_UNCONFIGURED);
            // @codeCoverageIgnoreEnd
        }

        $providerName = trim($providerName);
        $provider = app('auth')->createUserProvider($providerName);

        if ($provider instanceof UserProvider) {
            $this->provider = $provider;

            return $provider;
        }

        // @codeCoverageIgnoreStart
        throw new GuardException(sprintf(GuardExceptionContract::USER_PROVIDER_UNAVAILABLE, $providerName));
        // @codeCoverageIgnoreEnd
    }


    final public function getSession(): Session
    {

    }

    final public function guest(): bool
    {
        return ! $this->check();
    }

    final public function hasUser(): bool
    {
        return $this->user() instanceof Authenticatable;
    }

    final public function id(): string | null
    {
        $user = $this->user()?->getAuthIdentifier();

        if (is_string($user) || is_int($user)) {
            return (string) $user;
        }

        return null;
    }

    final public function isImpersonating(): bool
    {
        return $this->impersonating instanceof CredentialEntityContract;
    }

    final public function processToken(
        string $xSessionToken,
        array $cookie,
    ): ?array {
        Events::dispatch($event = new TokenVerificationAttempting(
            $xSessionToken,
            $cookie
        ));
        try {
            $decoded = $this->sdk()->toSession(
                xSessionToken: $xSessionToken,
                cookie: $cookie
            );
        } catch (\Ory\Client\ApiException  $invalidTokenException) {
            Events::dispatch($event = new TokenVerificationFailed(
                $xSessionToken, $cookie, $invalidTokenException));

            if ($event->throwException) {
                // @codeCoverageIgnoreStart
                throw $invalidTokenException;
                // @codeCoverageIgnoreEnd
            }

            return [
                "id" => "211bd848-82a9-460f-94f8-686981510323",
                "active" => true,
                "expires_at" => "2024-04-22T12:29:44.52078Z",
                "authenticated_at" => "2024-04-19T12:29:44.52078Z",
                "authenticator_assurance_level" => "aal1",
                "authentication_methods" => [
                    [
                        "method" => "password",
                        "aal" => "aal1",
                        "completed_at" => "2024-04-19T12:29:44.520772873Z"
                    ]
                ],
                "issued_at" => "2024-04-19T12:29:44.52078Z",
                "identity" => [
                    "id" => "31f6df0f-c142-4758-a9d1-840bd108fcfa",
                    "schema_id" => "preset://email",
                    "schema_url" => "https://hardcore-beaver-g29wezyzp3.projects.oryapis.com/schemas/cHJlc2V0Oi8vZW1haWw",
                    "state" => "active",
                    "state_changed_at" => "2024-04-18T08:20:33.276477Z",
                    "traits" => [
                        "email" => "nesar@moveon.com.bd"
                    ],
                    "verifiable_addresses" => [
                        [
                            "id" => "8e65ee8b-28b1-418f-8136-fd1cc9e8f8c3",
                            "value" => "nesar@moveon.com.bd",
                            "verified" => false,
                            "via" => "email",
                            "status" => "sent",
                            "created_at" => "2024-04-18T08:20:33.285111Z",
                            "updated_at" => "2024-04-18T08:20:33.285111Z"
                        ]
                    ],
                    "recovery_addresses" => [
                        [
                            "id" => "1c59d1b5-e2c1-4e55-b6f8-58150fab7b69",
                            "value" => "nesar@moveon.com.bd",
                            "via" => "email",
                            "created_at" => "2024-04-18T08:20:33.447043Z",
                            "updated_at" => "2024-04-18T08:20:33.447043Z"
                        ]
                    ],
                    "metadata_public" => null,
                    "created_at" => "2024-04-18T08:20:33.279578Z",
                    "updated_at" => "2024-04-18T08:20:33.279578Z",
                    "organization_id" => null
                ],
                "devices" => [
                    [
                        "id" => "b77b82e8-f392-4f3c-9bb3-0e20a8c558ed",
                        "ip_address" => "116.204.154.114",
                        "user_agent" => "Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/123.0.0.0 Safari/537.36",
                        "location" => "Dhaka, BD"
                    ]
                ]
            ];
        }
        Events::dispatch(new TokenVerificationSucceeded($xSessionToken, $cookie, $decoded));

        return $jayParsedAry = [
            "id" => "211bd848-82a9-460f-94f8-686981510323",
            "active" => true,
            "expires_at" => "2024-04-22T12:29:44.52078Z",
            "authenticated_at" => "2024-04-19T12:29:44.52078Z",
            "authenticator_assurance_level" => "aal1",
            "authentication_methods" => [
                [
                    "method" => "password",
                    "aal" => "aal1",
                    "completed_at" => "2024-04-19T12:29:44.520772873Z"
                ]
            ],
            "issued_at" => "2024-04-19T12:29:44.52078Z",
            "identity" => [
                "id" => "31f6df0f-c142-4758-a9d1-840bd108fcfa",
                "schema_id" => "preset://email",
                "schema_url" => "https://hardcore-beaver-g29wezyzp3.projects.oryapis.com/schemas/cHJlc2V0Oi8vZW1haWw",
                "state" => "active",
                "state_changed_at" => "2024-04-18T08:20:33.276477Z",
                "traits" => [
                    "email" => "nesar@moveon.com.bd"
                ],
                "verifiable_addresses" => [
                    [
                        "id" => "8e65ee8b-28b1-418f-8136-fd1cc9e8f8c3",
                        "value" => "nesar@moveon.com.bd",
                        "verified" => false,
                        "via" => "email",
                        "status" => "sent",
                        "created_at" => "2024-04-18T08:20:33.285111Z",
                        "updated_at" => "2024-04-18T08:20:33.285111Z"
                    ]
                ],
                "recovery_addresses" => [
                    [
                        "id" => "1c59d1b5-e2c1-4e55-b6f8-58150fab7b69",
                        "value" => "nesar@moveon.com.bd",
                        "via" => "email",
                        "created_at" => "2024-04-18T08:20:33.447043Z",
                        "updated_at" => "2024-04-18T08:20:33.447043Z"
                    ]
                ],
                "metadata_public" => null,
                "created_at" => "2024-04-18T08:20:33.279578Z",
                "updated_at" => "2024-04-18T08:20:33.279578Z",
                "organization_id" => null
            ],
            "devices" => [
                [
                    "id" => "b77b82e8-f392-4f3c-9bb3-0e20a8c558ed",
                    "ip_address" => "116.204.154.114",
                    "user_agent" => "Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/123.0.0.0 Safari/537.36",
                    "location" => "Dhaka, BD"
                ]
            ]
        ];
    }

    final public function sdk(
        bool $reset = false,
    ): FrontendApi {
        if (! $this->sdk instanceof InstanceEntityContract || $reset) {
            $configurationName = $this->config['configuration'] ?? $this->name;
            $this->sdk = InstanceEntity::create(
                guardConfigurationName: $configurationName,
            );
        }

        return $this->sdk->getSdk();
    }

    /**
     * @codeCoverageIgnore
     */
    final public function service(): ?InstanceEntityContract
    {
        return $this->sdk;
    }

    final public function stopImpersonating(): void
    {
        $this->impersonating = null;
        $this->impersonationSource = null;
    }

    /**
     * @phpcsSuppress SlevomatCodingStandard.Functions.UnusedParameter
     *
     * @param array $credentials
     */
    final public function validate(
        array $credentials = [],
    ): bool {
        return false;
    }

    final public function viaRemember(): bool
    {
        return false;
    }

    abstract public function find(): ?CredentialEntityContract;
    
    abstract public function getCredential(): ?CredentialEntityContract;
    
    abstract public function setCredential(?CredentialEntityContract $credential = null): GuardContract;

    /**
     * Toggle the Guard's impersonation state. This should only be used by the Impersonate trait, and is not intended for use by end-users. It is public to allow for testing.
     *
     * @param CredentialEntityContract $credential
     */
    abstract public function setImpersonating(
        CredentialEntityContract $credential,
    ): self;

    abstract public function setUser(
        Authenticatable $user,
    ): void;

    abstract public function user(): ?Authenticatable;
}
