<?php

declare(strict_types=1);

namespace MoveOn\Ory\Entities;

use MoveOn\Ory\Bridges\{CacheBridge};
use MoveOn\Ory\Configuration\Configuration as ConfigConstant;
use MoveOn\Ory\Exceptions\ControllerException;
use MoveOn\Ory\Exceptions\GuardException;
use Ory\Client\Api\FrontendApi;
use Ory\Client\Configuration;
use Psr\Cache\CacheItemPoolInterface;

use function in_array;
use function is_array;
use function is_string;

/**
 * @api
 */
abstract class InstanceEntityAbstract extends EntityAbstract
{
    public function __construct(
        protected ?FrontendApi $mainInstance = null,
        protected ?FrontendApi $proxyInstance = null,
        protected $configuration = null,
        protected ?CacheItemPoolInterface $tokenCachePool = null,
        protected ?string $guardConfigurationKey = null,
    ) {
    }

    final public function getConfiguration()
    {
        if (! $this->configuration) {
            $configuration = [];

            $defaultConfiguration = config('oryFrontend.guards.default');
            if (is_array($defaultConfiguration) && [] !== $defaultConfiguration) {
                $configuration = array_merge($configuration, array_filter($defaultConfiguration));
            }

            $guardConfiguration = [];

            if (null !== $this->guardConfigurationKey && '' !== $this->guardConfigurationKey && 'default' !== $this->guardConfigurationKey) {
                $guardConfiguration = config('oryFrontend.guards.' . $this->guardConfigurationKey) ?? [];
            }

            if (is_array($defaultConfiguration) && [] !== $defaultConfiguration) {
                $configuration = array_merge($configuration, array_filter($defaultConfiguration));
            }

            if (is_array($guardConfiguration) && [] !== $guardConfiguration) {
                $configuration = array_merge($configuration, array_filter($guardConfiguration));
            }
            $configuration = $this->bootTokenCache($configuration);
            $this->configuration = $configuration;
        }

        return $this->configuration;
    }

    final public function setGuardConfigurationKey(
        ?string $guardConfigurationKey = null,
    ): self {
        $this->guardConfigurationKey = $guardConfigurationKey;

        return $this;
    }

    final public function getMainInstance(): FrontendApi
    {
        if (! $this->mainInstance instanceof FrontendApi) {
            $config = $this->getConfiguration();
            $host   = $config[ConfigConstant::CONFIG_ORY_MAIN_HOST] ?? null;

            if (!$host) {
                throw new GuardException(GuardException::PROVIDER_HOST_UNCONFIGURED);
            }
            $config = (new Configuration())->setHost($host);
            $this->mainInstance = new FrontendApi(config: $config);
        }
        return $this->mainInstance;
    }

    final public function getProxyInstance(): FrontendApi
    {
        if (! $this->proxyInstance instanceof FrontendApi) {
            $config = $this->getConfiguration();
            $host   = $config[ConfigConstant::CONFIG_ORY_PROXY_HOST] ?? null;

            if (!$host) {
                throw new GuardException(GuardException::PROVIDER_HOST_UNCONFIGURED);
            }
            $config = (new Configuration())->setHost($host);
            $this->proxyInstance = new FrontendApi(config: $config);
        }
        return $this->proxyInstance;
    }

    abstract public function reset(): self;

    /**
     * @param null|array<string> $configuration
     */
    abstract public function setConfiguration(
        array | null $configuration = null,
    ): self;

    protected function bootTokenCache(array $config): array
    {
        $tokenCache = $config['tokenCache'] ?? null;

        if (false === $tokenCache) {
            unset($config['tokenCache']);

            return $config;
        }

        if (null === $tokenCache) {
            $tokenCache = $this->getTokenCachePool();
        }

        if (is_string($tokenCache)) {
            $tokenCache = app(trim($tokenCache));
        }

        $config['tokenCache'] = $tokenCache instanceof CacheItemPoolInterface ? $tokenCache : null;

        return $config;
    }


    protected function getTokenCachePool(): CacheItemPoolInterface
    {
        if (! $this->tokenCachePool instanceof CacheItemPoolInterface) {
            $this->tokenCachePool = app(CacheBridge::class);
        }

        return $this->tokenCachePool;
    }
}
