<?php

declare(strict_types=1);

namespace MoveOn\Ory\Guards;

use MoveOn\Ory\Entities\{CredentialEntity, CredentialEntityContract};
use MoveOn\Ory\UserProviderContract;
use Illuminate\Contracts\Auth\Authenticatable;
use MoveOn\Ory\Utility\HttpResponse;

use function is_array;
use function is_string;

/**
 * Authorization guard for stateless token-based authentication.
 *
 * @api
 */
final class AuthorizationGuard extends GuardAbstract implements AuthorizationGuardContract
{
    public function find(): ?CredentialEntityContract
    {
        if ($this->isImpersonating()) {
            return $this->getImposter();
        }

        return $this->findToken();
    }

    public function findToken(): ?CredentialEntityContract
    {
        if ($this->isImpersonating()) {
            return $this->getImposter();
        }
        $cookie = app('request')->cookie();
        $sessionToken = trim(app('request')->header("X-Session-Token") ?? '');
        if (count($cookie) === 0 && '' === $sessionToken) {
            return null;
        }
        $sessionResponse = $this->processToken(
            xSessionToken: $sessionToken,
            cookie: $cookie
        );

        /**
         * @var null|array<string> $sessionResponse
         */
        if (null === $sessionResponse) {
            return null;
        }

        $provider = $this->getProvider();
        // @codeCoverageIgnoreStart
        if (! $provider instanceof UserProviderContract) {
            return null;
        }
        // @codeCoverageIgnoreEnd

        $user = $provider->getRepository()->fromAccessToken(
            session: $sessionResponse,
            config: $provider->config
        );
        
        // @codeCoverageIgnoreStart
        if (! $user instanceof Authenticatable) {
            return null;
        }

        $user->provider_session = $this->prepareSession($sessionResponse);
        // @codeCoverageIgnoreEnd
        return CredentialEntity::create(
            user: $user
        );
    }
    

    public function prepareSession($sessionResponse): array
    {
        return [
            "id"               => $sessionResponse["id"],
            "active"           => $sessionResponse["active"],
            "expires_at"       => $sessionResponse["expires_at"],
            "authenticated_at" => $sessionResponse["authenticated_at"],
            "issued_at"        => $sessionResponse["issued_at"]
        ];
    }

    public function getCredential(): ?CredentialEntityContract
    {
        if ($this->isImpersonating()) {
            return $this->getImposter();
        }

        return $this->credential;
    }


    public function setCredential(
        ?CredentialEntityContract $credential = null,
    ): self {
        $this->stopImpersonating();

        $this->credential = $credential;

        return $this;
    }

    /**
     * @param CredentialEntityContract $credential
     */
    public function setImpersonating(
        CredentialEntityContract $credential,
    ): self {
        $this->impersonationSource = self::SOURCE_TOKEN;
        $this->impersonating = $credential;

        return $this;
    }

    public function setUser(
        Authenticatable $user,
    ): void {
        if ($this->isImpersonating()) {
            if ($this->getImposter()?->getUser() === $user) {
                return;
            }

            $this->stopImpersonating();
        }

        $credential = $this->getCredential() ?? CredentialEntity::create();
        $credential->setUser($user);

        $this->setCredential($credential);
    }

    public function user(): ?Authenticatable
    {
        if ($this->isImpersonating()) {
            return $this->getImposter()?->getUser();
        }

        $currentUser = $this->getCredential()?->getUser();

        if ($currentUser instanceof Authenticatable) {
            return $currentUser;
        }

        $entity = $this->find();
        if ($entity instanceof CredentialEntityContract) {
            return $entity?->getUser();
        }

        return null;
    }
}
