<?php

declare(strict_types=1);

namespace MoveOn\Ory\Guards;

use MoveOn\Ory\Entities\{CredentialEntity, CredentialEntityContract};
use MoveOn\Ory\UserProviderContract;
use Illuminate\Contracts\Auth\Authenticatable;
use MoveOn\Ory\Utility\HttpResponse;

use function is_array;
use function is_string;

/**
 * Authorization guard for stateless token-based authentication.
 *
 * @api
 */
final class AuthorizationGuard extends GuardAbstract implements AuthorizationGuardContract
{
    public function find(): ?CredentialEntityContract
    {
        if ($this->isImpersonating()) {
            return $this->getImposter();
        }

        return $this->findToken();
    }

    public function findToken(): ?CredentialEntityContract
    {
        if ($this->isImpersonating()) {
            return $this->getImposter();
        }
        $cookie = app('request')->cookie();
        $sessionToken = trim(app('request')->header("X-Session-Token") ?? '');
        if (count($cookie) === 0 && '' === $sessionToken) {
            return null;
        }
        $sessionResponse = $this->processToken(
            xSessionToken: $sessionToken,
            cookie: $cookie
        );

        /**
         * @var null|array<string> $sessionResponse
         */
        if (null === $sessionResponse) {
            return null;
        }

        $provider = $this->getProvider();
        // @codeCoverageIgnoreStart
        if (! $provider instanceof UserProviderContract) {
            return null;
        }
        // @codeCoverageIgnoreEnd

        $user = $provider->getRepository()->fromAccessToken(
            session: $sessionResponse,
            config: $provider->config
        );
        // @codeCoverageIgnoreStart
        if (! $user instanceof Authenticatable) {
            $this->callMissingUserCallback(
                providerConfig: $provider->config,
                sessionResponse: $sessionResponse,
                sessionToken: $sessionToken,
                cookie: $cookie,
                guardName: $this->getName(),
            );
            $user = $provider->getRepository()->fromAccessToken(
                session: $sessionResponse,
                config: $provider->config
            );
            if (! $user instanceof Authenticatable) {
                return null;
            }
        }

        $user->provider_session = $this->prepareSession($sessionResponse);
        // @codeCoverageIgnoreEnd
        return CredentialEntity::create(
            user: $user
        );
    }
    

    public function prepareSession($sessionResponse): array
    {
        return [
            "id"               => $sessionResponse["id"],
            "active"           => $sessionResponse["active"],
            "expires_at"       => $sessionResponse["expires_at"],
            "authenticated_at" => $sessionResponse["authenticated_at"],
            "issued_at"        => $sessionResponse["issued_at"]
        ];
    }

    public function getCredential(): ?CredentialEntityContract
    {
        if ($this->isImpersonating()) {
            return $this->getImposter();
        }

        return $this->credential;
    }


    public function setCredential(
        ?CredentialEntityContract $credential = null,
    ): self {
        $this->stopImpersonating();

        $this->credential = $credential;

        return $this;
    }

    /**
     * @param CredentialEntityContract $credential
     */
    public function setImpersonating(
        CredentialEntityContract $credential,
    ): self {
        $this->impersonationSource = self::SOURCE_TOKEN;
        $this->impersonating = $credential;

        return $this;
    }

    public function setUser(
        Authenticatable $user,
    ): void {
        if ($this->isImpersonating()) {
            if ($this->getImposter()?->getUser() === $user) {
                return;
            }

            $this->stopImpersonating();
        }

        $credential = $this->getCredential() ?? CredentialEntity::create();
        $credential->setUser($user);

        $this->setCredential($credential);
    }

    public function user(): ?Authenticatable
    {
        if ($this->isImpersonating()) {
            return $this->getImposter()?->getUser();
        }

        $currentUser = $this->getCredential()?->getUser();

        if ($currentUser instanceof Authenticatable) {
            return $currentUser;
        }

        $entity = $this->find();
        if ($entity instanceof CredentialEntityContract) {
            return $entity?->getUser();
        }

        return null;
    }

    private function callMissingUserCallback(array $providerConfig, array $sessionResponse, ?string $sessionToken, array $cookie, ?string $guardName = null): void
    {
        $rawCallback = $providerConfig['on_user_missing'] ?? null;

        $resolved = $this->resolveCallback($rawCallback);
        if (!is_callable($resolved)) {
            return;
        }

        $this->invokeCallback($resolved, $sessionResponse, $providerConfig, $sessionToken, $cookie, $guardName);
    }

    private function resolveCallback(mixed $candidate): ?callable
    {
        if (is_callable($candidate)) {
            return $candidate;
        }

        if (is_array($candidate) && count($candidate) === 2) {
            $classOrObject = $candidate[0] ?? null;
            $methodName    = $candidate[1] ?? null;

            if (is_object($classOrObject) && is_string($methodName)) {
                return is_callable([$classOrObject, $methodName]) ? [$classOrObject, $methodName] : null;
            }

            if (is_string($classOrObject) && is_string($methodName)) {
                $instance = $this->makeIfPossible($classOrObject);
                if ($instance && is_callable([$instance, $methodName])) {
                    return [$instance, $methodName];
                }
                if (is_callable([$classOrObject, $methodName], true)) {
                    return [$classOrObject, $methodName];
                }
            }

            return null;
        }

        if (is_string($candidate) && false !== strpos($candidate, '@')) {
            [$className, $methodName] = explode('@', $candidate, 2);
            $instance = $this->makeIfPossible($className);
            if ($instance && is_callable([$instance, $methodName])) {
                return [$instance, $methodName];
            }
            if (is_callable([$className, $methodName], true)) {
                return [$className, $methodName];
            }

            return null;
        }

        if (is_string($candidate) && function_exists($candidate)) {
            return $candidate;
        }

        return null;
    }

    private function makeIfPossible(string $className): ?object
    {
        if (app()->bound($className)) {
            return app($className);
        }

        if (class_exists($className)) {
            return new $className();
        }

        return null;
    }

    private function invokeCallback(callable $callback, array $sessionResponse, array $providerConfig, ?string $sessionToken, array $cookie, ?string $guardName = null): void
    {
        $callback($sessionResponse, $providerConfig, $sessionToken, $cookie, $guardName);
    }
}
