<?php

namespace MoveOn\Subscription\Modules\Gateway;

use Exception;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use MoveOn\Subscription\Response\Gateway\Paypal\PaypalAuthResponse;

abstract class BasePaypalGatewayManager
{
    protected string $clientId;
    protected string $secret;

    public function __construct(string $clientId, string $secret)
    {
        $this->clientId = $clientId;
        $this->secret   = $secret;
    }

    protected function getToken(): PaypalAuthResponse
    {
        $token = Cache::get("moveon-subscription-paypal-token");
        if (empty($token)) {
            $authResponse = Http::asForm()
                ->withHeaders(
                    [
                        "Content-Type" => "application/x-www-form-urlencoded",
                    ]
                )
                ->withBasicAuth($this->clientId, $this->secret)
                ->post(
                    config("subscription.paypal.base_url") . "/oauth2/token",
                    [
                        "grant_type" => "client_credentials",
                    ]
                );
            if ($authResponse->status() == 200) {
                $token = new PaypalAuthResponse(...$authResponse->json());
                Cache::put('moveon-subscription-paypal-token', $token, $token->expires_in - 10);
                return $token;
            }

            throw new Exception(get_class($this) . ":getToken:Paypal Authentication Failed");
        }

        return $token;
    }
    protected function isSuccessResponse(int $status): bool
    {
        return $status == 200 || $status == 201 || $status == 202 || $status == 204;
    }

    protected function clientErrorResponse(int $status): bool
    {
        return $status >= 400 && $status < 500;
    }

    protected function isServerErrorResponse(int $status): bool
    {
        return $status >= 500 && $status < 599;
    }

    protected function route($url, array $params = [])
    {
        return config("subscription.paypal.base_url") . "/$url?" . Arr::query($params);
    }
}