<?php

namespace MoveOn\Subscription\Service;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use MoveOn\Subscription\Contracts\ServiceContractDiscount;
use MoveOn\Subscription\Enums\DiscountDuration;
use MoveOn\Subscription\Enums\FeeType;
use MoveOn\Subscription\Enums\Gateway;
use MoveOn\Subscription\Enums\GatewaySlug;
use MoveOn\Subscription\Models\Discount;
use MoveOn\Subscription\Models\DiscountUsageHistory;
use MoveOn\Subscription\Models\DiscountUser;
use MoveOn\Subscription\Models\PaymentGateway;
use MoveOn\Subscription\QueryFilters\DiscountFilter;
use MoveOn\Subscription\QueryFilters\DiscountUsageHistoryFilter;
use MoveOn\Subscription\QueryFilters\DiscountUserFilter;
use MoveOn\Subscription\Requests\DiscountStoreRequest;
use MoveOn\Subscription\Requests\DiscountUpdateRequest;
use MoveOn\Common\Traits\Makeable;
use Stripe\StripeClient;

class DiscountService implements ServiceContractDiscount
{
    use Makeable;

    /**
     * List all discounts with filters
     * @param Request $request
     * @return array
     */
    public function listDiscount(Request $request)
    : array
    {
        $perPage  = $request->filled("per_page") ? min([$request->get("per_page"), 100]) : 10;
        $gateways = Discount::filter(DiscountFilter::class, $request)->paginate($perPage);

        return [
            "result"  => $gateways,
            "filters" => DiscountFilter::filterableDetails(),
        ];
    }

    /**
     * Create discount
     * @param DiscountStoreRequest $request
     * @param int $gatewayId
     * @return Discount
     */
    public function createDiscount(DiscountStoreRequest $request, int $gatewayId)
    : Discount
    {
        return DB::transaction(function () use ($request, $gatewayId) {
            $form               = $request->toArray();
            $form["duration"]   = $form["duration"]->value;
            $gateway            = PaymentGateway::findOrFail($gatewayId);
            $form["gateway_id"] = $gatewayId;
            $discount           = Discount::create($form);

            if ($gateway->slug == GatewaySlug::STRIPE()) {
                $stripe     = new StripeClient(config("subscription.stripe.client_secret"));
                $stripeForm = [
                    'name'     => $form["name"],
                    'currency' => $form["currency"],
                    'duration' => $form["duration"],
                ];

                if ($form["amount_type"] == FeeType::PERCENTAGE()) {
                    $stripeForm["percent_off"] = $form["amount"];
                } else {
                    $stripeForm["amount_off"] = $form["amount"];
                }

                if ($form["duration"] == DiscountDuration::REPEATING()) {
                    $stripeForm["duration_in_months"] = $form["duration_in_months"];
                }
                $coupon = $stripe->coupons->create($stripeForm);

                $discount->createMeta("STRIPE_COUPON_ID", $coupon->id);

            }

            return $discount;
        });
    }

    /**
     * Update discount
     * @param Discount $discount
     * @param DiscountUpdateRequest $request
     * @return Discount
     * @throws \Stripe\Exception\ApiErrorException
     */
    public function updateDiscount(Discount $discount, DiscountUpdateRequest $request)
    : Discount
    {
        $form = $request->toArray();
        $discount->update($form);

        $stripe = new StripeClient(config("subscription.stripe.client_secret"));
        $stripe->coupons->update(
            $discount->getMeta("STRIPE_COUPON_ID"),
            ['name' => $form["name"]]
        );

        return $discount;
    }

    /**
     * Activate discount
     * @param Discount $discount
     * @return bool
     */
    public function activateDiscount(Discount $discount)
    : bool
    {
        return $discount->update(["is_active" => true]);
    }

    /**
     * Deactivate discount
     * @param Discount $discount
     * @return bool
     */
    public function deactivateDiscount(Discount $discount)
    : bool
    {
        return $discount->update(["is_active" => false]);
    }

    /**
     * Delete discount
     * @param Discount $discount
     * @return bool
     */
    public function deleteDiscount(Discount $discount)
    : bool
    {
        DB::transaction(function () use ($discount) {
            $stripeCouponId = $discount->getMeta("STRIPE_COUPON_ID");
            $discount->delete();

            $stripe = new StripeClient(config("subscription.stripe.client_secret"));
            $stripe->coupons->delete($stripeCouponId, []);
        });

        return true;
    }

    /**
     * List discount users
     * @param Request $request
     * @return array
     */
    public function listDiscountUser(Request $request)
    : array
    {
        $perPage = $request->filled("per_page") ? min([$request->get("per_page"), 100]) : 10;

        $discountUsers = DiscountUser::filter(DiscountUserFilter::class)
                                     ->paginate($perPage)
        ;

        return [
            "result"  => $discountUsers,
            "filters" => DiscountUserFilter::filterableDetails(),
        ];
    }


    /**
     * Create discount user
     * @param Discount $discount
     * @param string|null $owner_type
     * @param int|null $owner_id
     * @param int|null $gateway_id
     * @param int|null $plan_id
     * @return DiscountUser
     */
    public function createDiscountUser(
        Discount $discount,
        ?string  $owner_type = null,
        ?int     $owner_id = null,
        ?int     $gateway_id = null,
        ?int     $plan_id = null
    )
    : DiscountUser
    {
        return DiscountUser::create(
            [
                "discount_id" => $discount->id,
                "owner_type"  => $owner_type,
                "owner_id"    => $owner_id,
                "gateway_id"  => $gateway_id,
                "plan_id"     => $plan_id,
            ]
        );
    }

    /**
     * Check discoutn user
     * @param Discount $discount
     * @param string|null $ownerType
     * @param int|null $ownerId
     * @param int|null $gatewayId
     * @param int $planId
     * @return bool
     */
    public function checkDiscountUser(
        Discount $discount,
        ?string  $ownerType,
        ?int     $ownerId,
        ?int     $gatewayId,
        int      $planId
    )
    : bool
    {
        if (DiscountUsageHistory::where('discount_id', $discount->id)->where('owner_type', $ownerType)->where(
            'owner_id',
            $ownerId
        )->where('gateway_id', $gatewayId)->where('plan_id', $planId)->exists()) {
            return true;
        }
        if (DiscountUsageHistory::where('discount_id', $discount->id)->where('owner_type', $ownerType)->where(
            'owner_id',
            $ownerId
        )->where('gateway_id', $gatewayId)->whereNull('plan_id')->exists()) {
            return true;
        }
        if (DiscountUsageHistory::where('discount_id', $discount->id)->where('owner_type', $ownerType)->where(
            'owner_id',
            $ownerId
        )->whereNull('gateway_id')->whereNull('plan_id')->exists()) {
            return true;
        }
        if (DiscountUsageHistory::where('discount_id', $discount->id)->whereNull('owner_type')->whereNull(
            'owner_id'
        )->whereNull('gateway_id')->whereNull('plan_id')->exists()) {
            return true;
        }

        return false;
    }

    /**
     * Delete discount user
     * @param DiscountUser $discountUser
     * @return bool
     */
    public function deleteDiscountUser(DiscountUser $discountUser)
    : bool
    {
        return $discountUser->delete();
    }

    /**
     * Activate discount user
     * @param DiscountUser $discountUser
     * @return bool
     */
    public function activateDiscountUser(DiscountUser $discountUser)
    : bool
    {
        return $discountUser->update(
            [
                "is_active" => true,
            ]
        );
    }

    /**
     * Deactivate discount user
     * @param DiscountUser $discountUser
     * @return bool
     */
    public function deactivateDiscountUser(DiscountUser $discountUser)
    : bool
    {
        return $discountUser->update(
            [
                "is_active" => false,
            ]
        );
    }

    /**
     * List discount usage history
     * @param Request $request
     * @return array
     */
    public function listDiscountUsageHistory(Request $request)
    : array
    {
        $perPage = $request->filled("per_page") ? min([$request->get("per_page"), 100]) : 10;

        $histories = DiscountUsageHistory::filter(DiscountUsageHistoryFilter::class, $request)
                                         ->with(["owner", "discount", "subscription"])
                                         ->paginate($perPage)
        ;

        return [
            "result"  => $histories,
            "filters" => DiscountUserFilter::filterableDetails(),
        ];
    }

    /**
     * Create discount usage history
     * @param Discount $discount
     * @param string $owner_type
     * @param int $owner_id
     * @param int $subscription_id
     * @return DiscountUsageHistory
     */
    public function createDiscountUsageHistory(
        Discount $discount,
        string   $owner_type,
        int      $owner_id,
        int      $subscription_id,
    )
    : DiscountUsageHistory
    {
        return DiscountUsageHistory::create(
            [
                "discount_id"     => $discount->id,
                "owner_type"      => $owner_type,
                "owner_id"        => $owner_id,
                "subscription_id" => $subscription_id,
            ]
        );
    }

    /**
     * Update discount usage history
     * @param DiscountUsageHistory $history
     * @param Discount $discount
     * @param string $owner_type
     * @param int $owner_id
     * @param int $subscription_id
     * @return bool
     */
    public function updateDiscountUsageHistory(
        DiscountUsageHistory $history,
        Discount             $discount,
        string               $owner_type,
        int                  $owner_id,
        int                  $subscription_id,
    )
    : bool
    {
        return $history->update(
            [
                "discount_id"     => $discount->id,
                "owner_type"      => $owner_type,
                "owner_id"        => $owner_id,
                "subscription_id" => $subscription_id,
            ]
        );
    }
}
