<?php

namespace MoveOn\Subscription\Service;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use MoveOn\Common\Traits\Makeable;
use MoveOn\Subscription\Collection\Collection;
use MoveOn\Subscription\Collection\Modules\Paypal\PaypalPlanBillingCycleCollection;
use MoveOn\Subscription\Collection\Modules\Paypal\PaypalPlanTierCollection;
use MoveOn\Subscription\Contracts\GatewayClientPaypal;
use MoveOn\Subscription\Contracts\GatewayClientStripe;
use MoveOn\Subscription\Contracts\ServiceContractPlan;
use MoveOn\Subscription\Enums\Gateway\Paypal\PaypalPlanStatus;
use MoveOn\Subscription\Enums\Gateway\Paypal\PlanTenureType;
use MoveOn\Subscription\Enums\Gateway\Paypal\PricingModel;
use MoveOn\Subscription\Enums\GatewaySlug;
use MoveOn\Subscription\Enums\IntervalUnit;
use MoveOn\Subscription\Enums\PricingScheme;
use MoveOn\Subscription\Enums\QuantitySource;
use MoveOn\Subscription\Enums\UsageType;
use MoveOn\Subscription\Models\PaymentGateway;
use MoveOn\Subscription\Models\Plan;
use MoveOn\Subscription\QueryFilters\PlanFilter;
use MoveOn\Subscription\Requests\Modules\Paypal\PaypalMoney;
use MoveOn\Subscription\Requests\Modules\Paypal\PaypalPaymentPreference;
use MoveOn\Subscription\Requests\Modules\Paypal\PaypalPricingScheme;
use MoveOn\Subscription\Requests\Modules\Paypal\Plan\PaypalPlanBillingCycle;
use MoveOn\Subscription\Requests\Modules\Paypal\Plan\PaypalPlanTier;
use MoveOn\Subscription\Requests\Modules\Paypal\Plan\PlanFrequency;
use MoveOn\Subscription\Requests\Modules\Paypal\Plan\PaypalPlanStore;
use MoveOn\Subscription\Requests\PlanCreateDTORequest;

class PlanService implements ServiceContractPlan
{
    use Makeable;

    private PaymentGateway $gateway;

    public function listPlans(Request $request)
    {
        $perPage   = $request->filled("per_page") ? min([$request->get("per_page"), 100]) : 10;
        $gateways = Plan::filter(PlanFilter::class, $request)->paginate($perPage);

        return [
            "result"  => $gateways,
            "filters" => PlanFilter::filterableDetails(),
        ];
    }

    public function createPlan(PlanCreateDTORequest $request)
    {
        $form    = $request->toArray();
        $product = $form["product"];
        $gateway = $product->gateway;
        $this->gateway = $gateway;
        unset($form["product"]);

        return DB::transaction(function () use ($product, $gateway, $form) {
            $tiers = $form["tiers"];
            unset($form["tiers"]);
            if ($gateway->slug == GatewaySlug::STRIPE()) {
                if (is_int($form["unit_amount"])) {
//                if (ceil($form["unit_amount"]) - floor($form["unit_amount"]) > 0) {
                    throw new \Exception("PlanService: unit must be integer", 422);
                }

                $form["interval_unit"]   = $form["interval_unit"]->value;
                $form["usage_type"]      = $form["usage_type"]->value;
                $form["pricing_scheme"]  = $form["pricing_scheme"]->value;
                $form["quantity_source"] = $form["quantity_source"]->value;

                $gatewayPlanForm = [
                    "currency"       => $form["currency"],
                    "product"        => $product->gateway_product_id,
                    "amount"         => $form["unit_amount"],
                    "active"         => $form["is_active"],
                    "interval"       => $form["interval_unit"],
                    "billing_scheme" => $form["pricing_scheme"] == PricingScheme::FIXED(
                    ) ? "per_unit" : PricingScheme::TIERED(),

                ];

                if ($form["pricing_scheme"] == PricingScheme::TIERED()) {
                    unset($gatewayPlanForm["amount"]);
                    $tiers = $tiers->items();
                    if (end($tiers)["end"] != null) {
                        throw new \Exception("Tier last value should be empty", 422);
                    }
                    $newTiers = [];
                    foreach ($tiers as $tier) {
                        $newTiers[] = [
                            "unit_amount_decimal" => $tier["price"],
                            "up_to"               => $tier["end"],
                        ];
                    }
                    $newTiers[count($newTiers) - 1]["up_to"] = "inf";

                    $gatewayPlanForm["tiers_mode"]     = "graduated";
                    $gatewayPlanForm["billing_scheme"] = PricingScheme::TIERED();
                    $gatewayPlanForm["tiers"]          = $newTiers;
                }

                $stripe      = app(GatewayClientStripe::class);
                $gatewayPlan = $stripe->plan()->createPlan($gatewayPlanForm);

                $form["gateway_id"]      = $gateway->id;
                $form["product_id"]      = $product->id;
                $form["gateway_plan_id"] = $gatewayPlan->id;
                $plan                    = Plan::create($form);
                $newTiers                = [];
                foreach ($tiers as $tier) {
                    $newTiers[] = [
                        "plan_id"     => $plan->id,
                        "start"       => $tier["start"],
                        "end"         => $tier["end"] ?? null,
                        "unit_amount" => $tier["price"],
                    ];
                }
                $plan->planTiers()->createMany($newTiers);
                return $plan;
            }


            if ($gateway->slug == GatewaySlug::PAYPAL()) {
                $paypalBillingCycle = PaypalPlanBillingCycleCollection::make();

                if ($form["trial_period_days"] > 0) {
                    [$paypalPricingScheme, $paypalPlanFrequency] = $this->generatePaypalTrialBillingCycleRequest($form);

                    $paypalBillingCycle = $paypalBillingCycle->add(
                        new PaypalPlanBillingCycle(
                            $paypalPlanFrequency,
                            1,
                            PlanTenureType::TRIAL,
                            $paypalPricingScheme,
                            1
                        )
                    );
                }

                $tiers = $this->generatePaypalTiersFromRequest($tiers, "USD");
                [$paypalPricingScheme, $paypalPlanFrequency] = $this->generatePaypalRegularBillingCycleRequest(
                    $form,
                    $tiers
                );

                $paypalBillingCycle = $paypalBillingCycle->add(
                    new PaypalPlanBillingCycle(
                        $paypalPlanFrequency,
                        $form["trial_period_days"] > 0 ? 2 : 1,
                        PlanTenureType::REGULAR,
                        $paypalPricingScheme,
                        1
                    )
                );

                $paypalPaymentPreference = $this->generatePaypalPaymentPreferenceRequest($form);

                $gatewayPlanForm = new PaypalPlanStore(
                    $form["name"],
                    $paypalBillingCycle,
                    $paypalPaymentPreference,
                    $product->gateway_product_id,
                    $form["is_active"] ? PaypalPlanStatus::ACTIVE : PaypalPlanStatus::INACTIVE,
                    true,
                );

                $paypalClient            = app(GatewayClientPaypal::class);
                $gatewayPlan             = $paypalClient->plan()->createPlan($gatewayPlanForm);
                $form["gateway_plan_id"] = $gatewayPlan->id;
                $form["interval_unit"]   = $form["interval_unit"]->value;
                $form["usage_type"]      = $form["usage_type"]->value;
                $form["pricing_scheme"]  = $form["pricing_scheme"]->value;
                $form["quantity_source"] = $form["quantity_source"]->value;
                $form["gateway_id"]      = $product->gateway_id;
                $form["product_id"]      = $product->id;

                $plan     = Plan::create($form);
                $tiers    = $tiers->pureArray();
                $newTiers = [];
                foreach ($tiers as $tier) {
                    $newTiers[] = [
                        "plan_id"     => $plan->id,
                        "start"       => $tier["starting_quantity"],
                        "end"         => $tier["ending_quantity"] ?? null,
                        "unit_amount" => $tier["amount"]["value"],
                    ];
                }

                $plan->planTiers()->createMany($newTiers);
                return $plan;
            }

            throw new \Exception("Unknown Gateway Given", 422);
        });
    }

    private function generatePaypalPaymentPreferenceRequest(array $form): PaypalPaymentPreference
    {
        $paypalSetupFee = new PaypalMoney(
            $form["currency"],
            max($form["system_usage_charge"], 10)
        );

        $autoBillOutstanding = $this->gateway->getMeta("PAYPAL_PLAN_AUTO_BILL_OUTSTANDING")??true;
        $setupFeeFailureThreshold = $this->gateway->getMeta("PAYPAL_PAYMENT_FAILURE_THRESHOLD")??5;
        $setupFeeFailureAction = $this->gateway->getMeta("PAYPAL_SETUP_FEE_FAILURE_ACTION")??"CONTINUE";

        return new PaypalPaymentPreference(
            $autoBillOutstanding,
            $setupFeeFailureAction,
            $setupFeeFailureThreshold,
            $paypalSetupFee
        );
    }

    private function generatePaypalTrialBillingCycleRequest(array $form): array
    {
        $paypalPlanFrequency = new PlanFrequency(
            IntervalUnit::PAYPAL_DAY,
            $form["trial_period_days"],
        );

        $paypalPricingScheme = null;

        return [$paypalPricingScheme, $paypalPlanFrequency];
    }

    private function generatePaypalRegularBillingCycleRequest(array $form, $tiers): array
    {
        $paypalPricingScheme = new PaypalPricingScheme(
            null,
            PricingModel::TIERED,
            $tiers
        );

        $paypalPlanFrequency = new PlanFrequency(
            $form["interval_unit"],
            $form["interval_count"],
        );

        return [$paypalPricingScheme, $paypalPlanFrequency];
    }

    public function activate(Plan $plan)
    {
        $gateway = $plan->gateway;

        if ($gateway->slug == GatewaySlug::STRIPE()) {
            $stripe = app(GatewayClientStripe::class);
            $stripe->plan()->updatePlan(
                $plan->gateway_plan_id,
                [
                    "active" => true,
                ]
            );
            return $plan->update(
                [
                    "is_active" => true,
                ]
            );
        }

        if ($gateway->slug == GatewaySlug::PAYPAL()) {
            $paypalClient = app(GatewayClientPaypal::class);
            $paypalClient->plan()->activatePlan($plan->gateway_plan_id);
            return $plan->update(
                [
                    "is_active" => true,
                ]
            );
        }

        return false;
    }


    public function deactivate(Plan $plan)
    {
        $gateway = $plan->gateway;

        if ($gateway->slug == GatewaySlug::STRIPE()) {
            $stripe = app(GatewayClientStripe::class);
            $stripe->plan()->updatePlan(
                $plan->gateway_plan_id,
                [
                    "active" => false,
                ]
            );
            return $plan->update(
                [
                    "is_active" => false,
                ]
            );
        }

        if ($gateway->slug == GatewaySlug::PAYPAL()) {
            $paypalClient = app(GatewayClientPaypal::class);
            $paypalClient->plan()->deactivatePlan($plan->gateway_plan_id);
            return $plan->update(
                [
                    "is_active" => false,
                ]
            );
        }

        return false;
    }

    private function generatePaypalTiersFromRequest(Collection $tiers, string $currency): PaypalPlanTierCollection
    {
        $tiers      = $tiers->items();
        $paypalTier = PaypalPlanTierCollection::make();

        foreach ($tiers as $index => $value) {
            $tiers[$index] = $paypalTier->add(
                new PaypalPlanTier($value["start"], $value["end"], new PaypalMoney($currency, $value["price"]))
            );
        }

        return $paypalTier;
    }
}