<?php

namespace MoveOn\Subscription\Tests\Unit;


use Faker\Factory;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use MoveOn\Subscription\Enums\GatewaySlug;
use MoveOn\Subscription\Models\PaymentGateway;
use MoveOn\Subscription\Requests\PaymentGatewayRequest;
use MoveOn\Subscription\Service\PaymentGatewayService;
use MoveOn\Subscription\Tests\TestCase;

class PaymentGatewayServiceTest extends TestCase
{
    use RefreshDatabase;

    protected function setUp(): void
    {
        $this->setUpTheTestEnvironment();
    }

    public function test_payment_gateway_service_list()
    {
        $service = new PaymentGatewayService();
        $request = new Request();
        $list = $service->list($request);
        $this->assertIsArray($list);
        $this->assertArrayHasKey("result", $list);
        $this->assertArrayHasKey("filters", $list);
    }

    public function test_payment_gateway_service_store()
    {
        $faker = Factory::create();
        $name = $faker->name;
        $gatewayRequest = [
            "name" => $name,
            "slug" => Str::slug($name),
            "gateway_type" => Str::slug($name),
            "fee_type" => $faker->randomElement(["fixed", "percentage"]),
            "fee" => $faker->randomFloat(2, 1, 10),
            "logo" => $faker->url,
            "url" => $faker->url,
        ];
        $paymentGatewayRequest = new PaymentGatewayRequest(...$gatewayRequest);

        $service = new PaymentGatewayService();
        $gateway = $service->create($paymentGatewayRequest);

        $this->assertTrue(get_class($gateway) == PaymentGateway::class);
        $this->assertDatabaseHas(PaymentGateway::class, ["id" => $gateway->id]);
    }

    public function test_payment_gateway_service_update()
    {
        $faker = Factory::create();
        $name = $faker->name;
        $gatewayRequest = [
            "name" => $name,
            "slug" => GatewaySlug::STRIPE(),
            "gateway_type" => Str::slug($name),
            "fee_type" => $faker->randomElement(["fixed", "percentage"]),
            "fee" => $faker->randomFloat(2, 1, 10),
            "logo" => $faker->url,
            "url" => $faker->url,
        ];
        $paymentGatewayRequest = new PaymentGatewayRequest(...$gatewayRequest);

        $service = new PaymentGatewayService();
        $createdGateway = $service->create($paymentGatewayRequest);

        $gatewayRequest = [
            ...$createdGateway->toArray(),
            "name" => $faker->name,
        ];
        unset($gatewayRequest["created_at"]);
        unset($gatewayRequest["updated_at"]);
        unset($gatewayRequest["deleted_at"]);
        unset($gatewayRequest["id"]);
        $paymentGatewayRequest = new PaymentGatewayRequest(...$gatewayRequest);
        $updatedGateway = $service->update($createdGateway, $paymentGatewayRequest);

        $this->assertTrue($updatedGateway);
        $this->assertDatabaseHas(PaymentGateway::class, ["id" => $createdGateway->id, "name" => $gatewayRequest["name"]]);
    }

    public function test_payment_gateway_service_delete()
    {
        $faker = Factory::create();
        $name = $faker->name;
        $gatewayRequest = [
            "name" => $name,
            "slug" => GatewaySlug::STRIPE(),
            "gateway_type" => Str::slug($name),
            "fee_type" => $faker->randomElement(["fixed", "percentage"]),
            "fee" => $faker->randomFloat(2, 1, 10),
            "logo" => $faker->url,
            "url" => $faker->url,
        ];
        $paymentGatewayRequest = new PaymentGatewayRequest(...$gatewayRequest);

        $service = new PaymentGatewayService();
        $gateway = $service->create($paymentGatewayRequest);
        $isDeleted = $service->delete($gateway);

        $this->assertTrue($isDeleted);
    }


    public function test_payment_gateway_service_enable()
    {
        $faker = Factory::create();
        $name = $faker->name;
        $gatewayRequest = [
            "name" => $name,
            "slug" => Str::slug($name),
            "gateway_type" => Str::slug($name),
            "fee_type" => $faker->randomElement(["fixed", "percentage"]),
            "fee" => $faker->randomFloat(2, 1, 10),
            "logo" => $faker->url,
            "url" => $faker->url,
            "is_active" => false,
        ];
        $paymentGatewayRequest = new PaymentGatewayRequest(...$gatewayRequest);

        $service = new PaymentGatewayService();
        $gateway = $service->create($paymentGatewayRequest);
        $isEnabled = $service->enable($gateway);

        $this->assertTrue($isEnabled);
        $this->assertDatabaseHas(PaymentGateway::class, ["id" => $gateway->id, "is_active" => true]);
    }

    public function test_payment_gateway_service_disable()
    {
        $faker = Factory::create();
        $name = $faker->name;
        $gatewayRequest = [
            "name" => $name,
            "slug" => Str::slug($name),
            "gateway_type" => Str::slug($name),
            "fee_type" => $faker->randomElement(["fixed", "percentage"]),
            "fee" => $faker->randomFloat(2, 1, 10),
            "logo" => $faker->url,
            "url" => $faker->url,
            "is_active" => true,
        ];
        $paymentGatewayRequest = new PaymentGatewayRequest(...$gatewayRequest);

        $service = new PaymentGatewayService();
        $gateway = $service->create($paymentGatewayRequest);
        $isEnabled = $service->disable($gateway);

        $this->assertTrue($isEnabled);
        $this->assertDatabaseHas(PaymentGateway::class, ["id" => $gateway->id, "is_active" => false]);
    }
}