<?php

namespace MoveOn\Subscription\Tests\Unit;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\Request;
use Mockery;
use Mockery\MockInterface;
use MoveOn\Subscription\Collection\Modules\Paypal\PlanTierCollection;
use MoveOn\Subscription\Contracts\GatewayClientPaypal;
use MoveOn\Subscription\Contracts\GatewayClientStripe;
use MoveOn\Subscription\Database\Factories\PlanFactory;
use MoveOn\Subscription\Database\Factories\ProductFactory;
use MoveOn\Subscription\Enums\GatewaySlug;
use MoveOn\Subscription\Enums\IntervalUnit;
use MoveOn\Subscription\Enums\PricingScheme;
use MoveOn\Subscription\Enums\QuantitySource;
use MoveOn\Subscription\Enums\UsageType;
use MoveOn\Subscription\Models\PaymentGateway;
use MoveOn\Subscription\Models\Plan;
use MoveOn\Subscription\Requests\PlanCreateDTORequest;
use MoveOn\Subscription\Requests\PlanTierDTORequest;
use MoveOn\Subscription\Response\Gateway\Paypal\PaypalPlanResponse;
use MoveOn\Subscription\Service\PlanService;
use MoveOn\Subscription\Tests\TestCase;

class PlanServiceTest extends TestCase
{
    use RefreshDatabase;

    public function test_plan_list_gives_200()
    {
        $request = new Request();
        $list    = PlanService::make()->listPlans($request);
        $this->assertIsArray($list);
        $this->assertArrayHasKey("result", $list);
        $this->assertArrayHasKey("filters", $list);
    }

    private function initStripePlanCreate()
    {
        $gateway     = PaymentGateway::where("slug", GatewaySlug::STRIPE())->first();
        $product     = ProductFactory::new()->create(
            [
                "gateway_id" => $gateway->id,
            ]
        );
        $planRequest = PlanFactory::new()->make(
            [
                "gateway_id" => $gateway->id,
                "product_id" => $product->id,
            ]
        )->toArray();
        unset($planRequest["product_id"], $planRequest["gateway_id"]);
        $planRequest["product"]         = $product;
        $planRequest["pricing_scheme"]  = PricingScheme::from($planRequest["pricing_scheme"]);
        $planRequest["interval_unit"]   = IntervalUnit::from($planRequest["interval_unit"]);
        $planRequest["usage_type"]      = UsageType::from($planRequest["usage_type"]);
        $planRequest["quantity_source"] = QuantitySource::from($planRequest["quantity_source"]);
        $planRequest["unit_amount"]     = floor($planRequest["unit_amount"]);
        $planRequest["tiers"]           = PlanTierCollection::make()
            ->add(
                new PlanTierDTORequest(1, 10, 10)
            )
            ->add(
                new PlanTierDTORequest(11, null, 15)
            );

        unset($planRequest["gateway_plan_id"]);
        $request = new PlanCreateDTORequest(...$planRequest);

        $this->instance(
            GatewayClientStripe::class,
            Mockery::mock(GatewayClientStripe::class, function (MockInterface $mock) use ($request) {
                $id  = rand(11111, 99999);
                $res = json_decode(
                    <<<JSON
{
  "id": "$id",
  "object": "plan",
  "active": true,
  "aggregate_usage": null,
  "amount": 6900,
  "amount_decimal": "6900",
  "billing_scheme": "{$request->pricing_scheme->value}",
  "created": 1656487808,
  "currency": "usd",
  "interval": "{$request->interval_unit->value}",
  "interval_count": "{$request->interval_count}",
  "livemode": false,
  "metadata": {},
  "nickname": null,
  "product": "{$request->product->gateway_product_id}",
  "tiers_mode": "{$request->pricing_scheme->value}",
  "transform_usage": null,
  "trial_period_days": "{$request->trial_period_days}",
  "usage_type": "{$request->usage_type->value}"
}
JSON
                );
                $mock->shouldReceive('plan->createPlan')->andReturn($res);
            })
        );

        return [$gateway, $product, $request];
    }

    private function createStripePlan()
    {
        [$gateway, $product, $request] = $this->initStripePlanCreate();

        return PlanService::make()->createPlan($request);
    }

    public function test_stripe_plan_create_gives_200()
    {
        $plan = $this->createStripePlan();

        $this->assertTrue(get_class($plan) == Plan::class);
        $this->assertDatabaseHas(Plan::class, [
            "id"               => $plan->id,
            "gateway_plan_id"  => $plan->gateway_plan_id,
            "product_id"       => $plan->product_id,
            "currency"         => $plan->currency,
            "name"             => $plan->name,
            "description"      => $plan->description,
            "unit_amount"      => $plan->unit_amount,
            "quantity_source"  => $plan->quantity_source,
            "default_quantity" => $plan->default_quantity,
            "usage_type"       => $plan->usage_type,
        ]);
    }

    private function initPaypalPlanCreate()
    {
        $gateway     = PaymentGateway::where("slug", GatewaySlug::PAYPAL())->first();
        $product     = ProductFactory::new()->create(
            [
                "gateway_id" => $gateway->id,
            ]
        );
        $planRequest = PlanFactory::new()->make(
            [
                "gateway_id" => $gateway->id,
                "product_id" => $product->id,
            ]
        )->toArray();
        unset($planRequest["product_id"], $planRequest["gateway_id"]);
        $planRequest["product"]           = $product;
        $planRequest["pricing_scheme"]    = PricingScheme::from($planRequest["pricing_scheme"]);
        $planRequest["interval_unit"]     = IntervalUnit::PAYPAL_DAY;
        $planRequest["trial_period_days"] = 10;
        $planRequest["usage_type"]        = UsageType::from($planRequest["usage_type"]);
        $planRequest["quantity_source"]   = QuantitySource::from($planRequest["quantity_source"]);
        $planRequest["unit_amount"]       = floor($planRequest["unit_amount"]);
        $planRequest["tiers"]             = PlanTierCollection::make()
            ->add(
                new PlanTierDTORequest(1, 10, 10)
            )
            ->add(
                new PlanTierDTORequest(11, null, 15)
            );

        unset($planRequest["gateway_plan_id"]);
        $request = new PlanCreateDTORequest(...$planRequest);

        $this->instance(
            GatewayClientPaypal::class,
            Mockery::mock(GatewayClientPaypal::class, function (MockInterface $mock) use ($request) {
                $id  = rand(11111, 99999);
                $res = new PaypalPlanResponse(
                    $id,
                    $request->product->gateway_product_id,
                    $request->name,
                    true,
                    $request->usage_type->value,
                    122222
                );
                $mock->shouldReceive('plan->createPlan')->andReturn($res);
            })
        );

        return [$gateway, $product, $request];
    }

    public function test_paypal_plan_create_gives_200()
    {
        [$gateway, $product, $request] = $this->initPaypalPlanCreate();

        $plan = PlanService::make()->createPlan($request);

        $this->assertTrue(get_class($plan) == Plan::class);
        $this->assertDatabaseHas(Plan::class, [
            "id"               => $plan->id,
            "gateway_plan_id"  => $plan->gateway_plan_id,
            "product_id"       => $plan->product_id,
            "currency"         => $plan->currency,
            "name"             => $plan->name,
            "description"      => $plan->description,
            "unit_amount"      => $plan->unit_amount,
            "quantity_source"  => $plan->quantity_source,
            "default_quantity" => $plan->default_quantity,
            "usage_type"       => $plan->usage_type,
        ]);
    }

    private function createPaypalPlan()
    {
        [$gateway, $product, $request] = $this->initPaypalPlanCreate();

        return PlanService::make()->createPlan($request);
    }

    private function initPaypalPlanActivate()
    {
        $this->instance(
            GatewayClientPaypal::class,
            Mockery::mock(GatewayClientPaypal::class, function (MockInterface $mock) {
                $mock->shouldReceive('plan->activatePlan')->andReturn(true);
            })
        );
    }


    private function initPaypalPlanDeactivate()
    {
        $this->instance(
            GatewayClientPaypal::class,
            Mockery::mock(GatewayClientPaypal::class, function (MockInterface $mock) {
                $mock->shouldReceive('plan->deactivatePlan')->andReturn(true);
            })
        );
    }

    private function initStripePlanActivate()
    {
        $this->instance(
            GatewayClientStripe::class,
            Mockery::mock(GatewayClientStripe::class, function (MockInterface $mock) {
                $mock->shouldReceive('plan->update')->andReturn(true);
            })
        );
    }


    private function initStripePlanDeactivate()
    {
        $this->instance(
            GatewayClientStripe::class,
            Mockery::mock(GatewayClientStripe::class, function (MockInterface $mock) {
                $mock->shouldReceive('plan->update')->andReturn(true);
            })
        );
    }

    public function test_paypal_plan_activate_200()
    {
        $plan = $this->createPaypalPlan();

        $this->initPaypalPlanActivate();

        $activatedPlan = PlanService::make()->activate($plan);

        $this->assertTrue($activatedPlan);
    }

    public function test_paypal_plan_deactivate_200()
    {
        $plan = $this->createPaypalPlan();

        $this->initPaypalPlanDeactivate();

        $deactivatedPlan = PlanService::make()->deactivate($plan);

        $this->assertTrue($deactivatedPlan);
    }

    public function test_stripe_plan_activate_200()
    {
        $plan = $this->createStripePlan();

        $this->initStripePlanActivate();

        $activatedPlan = PlanService::make()->activate($plan);

        $this->assertTrue($activatedPlan);
    }

    public function test_stripe_plan_deactivate_200()
    {
        $plan = $this->createStripePlan();

        $this->initStripePlanDeactivate();

        $activatedPlan = PlanService::make()->deactivate($plan);

        $this->assertTrue($activatedPlan);
    }

}