<?php

namespace MoveOn\Subscription\Modules\Gateway;

use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Http;
use MoveOn\Subscription\Contracts\PaypalPlanManagerContract;
use MoveOn\Subscription\Requests\Modules\Paypal\Plan\PaypalPlanStore;
use MoveOn\Subscription\Response\Gateway\Paypal\PaypalErrorResponse;
use MoveOn\Subscription\Response\Gateway\Paypal\PaypalPlanResponse;

class PaypalPlanManager extends BasePaypalGatewayManager implements PaypalPlanManagerContract
{
    /**
     * @param PaypalPlanStore $request
     * @return PaypalPlanResponse
     * @throws PaypalErrorResponse
     */
    public function createPlan(PaypalPlanStore $request): PaypalPlanResponse
    {
        $request = Arr::whereNotNull($request->pureArray(true));

        $response = Http::withHeaders(
            [
                "Content-Type" => "application/json",
                "Prefer" => "minimal",
            ]
        )
            ->withToken($this->getToken()->access_token)
            ->post(
                config("subscription.paypal.base_url") . "/billing/plans",
                $request
            );

        if ($this->isSuccessResponse($response->status())) {
            return new PaypalPlanResponse(...$response->json());
        }

        throw new PaypalErrorResponse(get_class($this) . ":createPlan", $response->json(), $response->status());
    }

    /**
     * @param string $planId
     * @return PaypalPlanResponse
     * @throws PaypalErrorResponse
     */
    public function detailsPlan(string $planId): PaypalPlanResponse
    {
        $response = Http::withHeaders(
            [
                "Content-Type" => "application/json",
            ]
        )
            ->withToken($this->getToken()->access_token)
            ->get(
                config("subscription.paypal.base_url") . "/billing/plans/$planId",
            );
        if ($this->isSuccessResponse($response->status())) return new PaypalPlanResponse(...$response->json());

        throw new PaypalErrorResponse(get_class($this) . ":detailsPlan", $response->json(), $response->status());
    }

    /**
     * @param string $planId
     * @return bool
     * @throws PaypalErrorResponse
     */
    public function activatePlan(string $planId): bool
    {
        $response = Http::withHeaders(
            [
                "Content-Type" => "application/json",
            ]
        )
            ->withToken($this->getToken()->access_token)
            ->post(
                config("subscription.paypal.base_url") . "/billing/plans/$planId/activate",
            );

        if ($this->isSuccessResponse($response->status())) return true;

        throw new PaypalErrorResponse(get_class($this) . ":activatePlan", $response->json(), $response->status());
    }

    /**
     * @param string $planId
     * @return bool
     * @throws PaypalErrorResponse
     */
    public function deactivatePlan(string $planId): bool
    {
        $response = Http::withHeaders(
            [
                "Content-Type" => "application/json",
            ]
        )
            ->withToken($this->getToken()->access_token)
            ->post(
                config("subscription.paypal.base_url") . "/billing/plans/$planId/deactivate",
            );

        if ($this->isSuccessResponse($response->status())) return true;

        throw new PaypalErrorResponse(get_class($this) . ":deactivatePlan", $response->json(), $response->status());
    }

}