<?php

namespace MoveOn\Subscription\Modules\Gateway;

use Illuminate\Support\Facades\Http;
use MoveOn\Subscription\Collection\Modules\Paypal\PaypalProductUpdateCollection;
use MoveOn\Subscription\Contracts\PaypalProductManagerContract;
use MoveOn\Subscription\Requests\Modules\Paypal\PaypalListRequest;
use MoveOn\Subscription\Requests\ProductStoreDTORequest;
use MoveOn\Subscription\Response\Gateway\Paypal\PaypalErrorResponse;
use MoveOn\Subscription\Response\Gateway\Paypal\Product\PaypalProductResponse;

class PaypalProductManager extends BasePaypalGatewayManager implements PaypalProductManagerContract
{
    /**
     * @param PaypalListRequest $request
     * @return array
     * @throws PaypalErrorResponse
     */
    public function listProduct(PaypalListRequest $request): array
    {
        $response = Http::withHeaders(
            [
                "Content-Type" => "application/json",
            ]
        )
            ->withToken($this->getToken()->access_token)
            ->get(
                $this->route(
                    "catalogs/products",
                    $request->toArray()
                )
            );

        if ($this->isSuccessResponse($response->status())) {
            $data = $response->json();
            return PaypalProductResponse::collection(
                $data["products"],
                $data["total_items"] ?? null,
                $data["total_pages"] ?? null,
                $data["links"] ?? []
            );
        }

        throw new PaypalErrorResponse(get_class($this) . ":listProduct", $response->json(), $response->status());
    }

    /**
     * @param ProductStoreDTORequest $request
     * @return PaypalProductResponse
     * @throws PaypalErrorResponse
     */
    public function createProduct(ProductStoreDTORequest $request): PaypalProductResponse
    {
        $response = Http::withHeaders(
            [
                "Content-Type" => "application/json",
            ]
        )
            ->withToken($this->getToken()->access_token)
            ->post(
                config("subscription.paypal.base_url") . "/catalogs/products",
                $request->toArray()
            );

        if ($this->isSuccessResponse($response->status())) {
            return new PaypalProductResponse(...$response->json());
        }

        throw new PaypalErrorResponse(get_class($this) . ":createProduct", $response->json(), $response->status());
    }

    /**
     * @param string $productId
     * @param PaypalProductUpdateCollection $collection
     * @return bool
     * @throws PaypalErrorResponse
     */
    public function updateProduct(string $productId, PaypalProductUpdateCollection $collection): bool
    {
        $collection = $collection->items();

        $response = Http::withHeaders(
            [
                "Content-Type" => "application/json",
            ]
        )
            ->withToken($this->getToken()->access_token)
            ->patch(
                config("subscription.paypal.base_url") . "/catalogs/products/$productId",
                $collection
            );

        if ($this->isSuccessResponse($response->status())) {
            return true;
        }

        throw new PaypalErrorResponse(get_class($this) . ":updateProduct", $response->json(), $response->status());
    }

    /**
     * @param string $productId
     * @return PaypalProductResponse
     * @throws PaypalErrorResponse
     */
    public function detailsProduct(string $productId): PaypalProductResponse
    {
        $response = Http::withHeaders(
            [
                "Content-Type" => "application/json",
            ]
        )
            ->withToken($this->getToken()->access_token)
            ->get(
                config("subscription.paypal.base_url") . "/catalogs/products/$productId",
            );

        if ($this->isSuccessResponse($response->status())) {
            return new PaypalProductResponse(...$response->json());
        }

        throw new PaypalErrorResponse(get_class($this) . ":listProduct", $response->json(), $response->status());
    }
}