<?php

namespace MoveOn\Subscription\Modules\Gateway;

use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Http;
use MoveOn\Subscription\Contracts\PaypalSubscriptionManagerContract;
use MoveOn\Subscription\Requests\Modules\Paypal\Subscription\PaypalSubscriptionStoreDTORequest;
use MoveOn\Subscription\Response\Gateway\Paypal\PaypalErrorResponse;
use MoveOn\Subscription\Response\Gateway\Paypal\PaypalSubscriptionResponse;

class PaypalSubscriptionManager extends BasePaypalGatewayManager implements PaypalSubscriptionManagerContract
{
    public function createSubscription(PaypalSubscriptionStoreDTORequest $request): PaypalSubscriptionResponse
    {
        $request = Arr::whereNotNull($request->pureArray(true));
        $response = Http::withHeaders(
            [
                "Content-Type" => "application/json",
                "Prefer" => "return=representation",
            ]
        )
            ->withToken($this->getToken()->access_token)
            ->post(
                config("subscription.paypal.base_url") . "/billing/subscriptions",
                $request
            );

        if ($this->isSuccessResponse($response->status())) {
            return new PaypalSubscriptionResponse(...$response->json());
        }

        throw new PaypalErrorResponse(get_class($this) . ":createSubscription", $response->json(), $response->status());
    }

    public function detailsSubscription(string $subscriptionId): PaypalSubscriptionResponse
    {
        $response = Http::withHeaders(
            [
                "Content-Type" => "application/json",
                "Prefer" => "return=representation",
            ]
        )
            ->withToken($this->getToken()->access_token)
            ->get(
                config("subscription.paypal.base_url") . "/billing/subscriptions/$subscriptionId",
            );

        if ($this->isSuccessResponse($response->status())) {
            return new PaypalSubscriptionResponse(...$response->json());
        }

        throw new PaypalErrorResponse(get_class($this) . ":detailsSubscription", $response->json(), $response->status());
    }

    public function cancelSubscription(string $subscriptionId): bool
    {
        $response = Http::withHeaders(
            [
                "Content-Type" => "application/json",
                "Prefer" => "minimal",
            ]
        )
            ->withToken($this->getToken()->access_token)
            ->post(
                config("subscription.paypal.base_url") . "/billing/subscriptions/$subscriptionId",
            );

        if ($this->isSuccessResponse($response->status())) {
            return true;
        }

        throw new PaypalErrorResponse(get_class($this) . ":cancelSubscription", $response->json(), $response->status());
    }
}