<?php

namespace MoveOn\Subscription\Service;

use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use MoveOn\Common\Traits\Makeable;
use MoveOn\Subscription\Contracts\GatewayClientPaypal;
use MoveOn\Subscription\Contracts\GatewayClientStripe;
use MoveOn\Subscription\Contracts\ServiceContractCustomer;
use MoveOn\Subscription\Enums\GatewaySlug;
use MoveOn\Subscription\Models\GatewayAssociatedCustomer;
use MoveOn\Subscription\Models\PaymentGateway;
use MoveOn\Subscription\Models\Product;
use MoveOn\Subscription\Modules\Gateway\ClientStripe;
use MoveOn\Subscription\QueryFilters\GatewayAssociatedCustomerFilter;
use MoveOn\Subscription\Requests\CustomerStoreDTORequest;
use Stripe\Collection;
use Stripe\Exception\ApiErrorException;

class GatewayAssociatedCustomerService implements ServiceContractCustomer
{
    use Makeable;

    public function listCustomer(Request $request)
    {
        $perPage   = $request->filled("per_page") ? min([$request->get("per_page"), 100]) : 10;

        $customers = GatewayAssociatedCustomer::filter(GatewayAssociatedCustomerFilter::class, $request)
            ->paginate($perPage);

        return [
            "result"  => $customers,
            "filters" => GatewayAssociatedCustomerFilter::filterableDetails(),
        ];
    }

    public function createCustomer(PaymentGateway $gateway, CustomerStoreDTORequest $request): ?GatewayAssociatedCustomer
    {
        /**
         * @var ClientStripe $stripe
         */
        if ($gateway->slug == GatewaySlug::STRIPE()) {
            $stripe                      = app(GatewayClientStripe::class);
            $customer                    = $stripe->customer()->createCustomer([
                "email" => $request->email,
                "name"  => $request->name
            ]);
            $form                        = $request->toArray();
            $form["gateway_customer_id"] = $customer->id;
            $form["gateway_id"]          = $gateway->id;
            return GatewayAssociatedCustomer::create($form);
        }

        if ($gateway->slug == GatewaySlug::PAYPAL()) {
            $form                        = $request->toArray();
            $form["gateway_customer_id"] = "N/A";
            $form["gateway_id"]          = $gateway->id;
            return GatewayAssociatedCustomer::create($form);
        }

        throw new \Exception("Unknown Payment Gateway Given");
    }

    public function activateCustomer(GatewayAssociatedCustomer $customer): bool
    {
        return $customer->update(
            [
                'is_active' => true,
            ]
        );
    }

    public function deactivateCustomer(GatewayAssociatedCustomer $customer): bool
    {
        return $customer->update(
            [
                'is_active' => false,
            ]
        );
    }

    /**
     * @throws ApiErrorException
     */
    public function getAllCustomerPaymentMethods(GatewayAssociatedCustomer $customer): JsonResponse | Collection
    {
        $gateway = $customer->gateway;
        if ($gateway->slug == GatewaySlug::STRIPE()) {
            /**
             * @var ClientStripe $stripe
             */
            $stripe = app(GatewayClientStripe::class);

            return $stripe->paymentMethod()->listPaymentMethods($customer->gateway_customer_id);
        } else {
            return response()->json([
                "success" => false,
                "message" => "Gateway not supported",
            ]);
        }

    }

    /**
     * @throws \Exception
     */
    public function setDefaultPaymentMethod(GatewayAssociatedCustomer $customer, string $paymentMethodId): bool
    {
        $gateway = $customer->gateway;
        if ($gateway->slug == GatewaySlug::STRIPE()) {
            /**
             * @var ClientStripe $stripe
             */
            $stripe = app(GatewayClientStripe::class);
            $stripe->customer()->setDefaultPaymentMethod([
                "customer_id"      => $customer->gateway_customer_id,
                "payment_method_id" => $paymentMethodId
            ]);
            return true;
        }

        if ($gateway->slug == GatewaySlug::PAYPAL()) {
            //TODO: Implement Paypal Default Payment Method IF NEEDED
            return true;
        }

        throw new \Exception("Unknown Payment Gateway Given");
    }

    /**
     * @throws ApiErrorException
     * @throws \Exception
     */
    public function detachPaymentMethod(GatewayAssociatedCustomer $customer, string $paymentMethodId): bool
    {
        $gateway = $customer->gateway;
        if ($gateway->slug == GatewaySlug::STRIPE()) {
            /**
             * @var ClientStripe $stripe
             */
            $stripe = app(GatewayClientStripe::class);
            $customerDetails = $stripe->customer()->retrieveCustomer([
                "customer_id" => $customer->gateway_customer_id
            ]);

            if ($customerDetails->invoice_settings->default_payment_method === $paymentMethodId) {
                throw new \Exception("You cannot detach the default payment method. Please set another payment method as default first.");
            }

            $stripe->paymentMethod()->detachPaymentMethod($paymentMethodId);
            return true;
        }

        if ($gateway->slug == GatewaySlug::PAYPAL()) {
            //TODO: Implement Paypal Default Payment Method IF NEEDED
            return true;
        }

        throw new \Exception("Unknown Payment Gateway Given");
    }
}