<?php

namespace MoveOn\Subscription\Tests\Unit;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\Request;
use MoveOn\Subscription\Database\Factories\DiscountFactory;
use MoveOn\Subscription\Database\Factories\DiscountUsageHistoryFactory;
use MoveOn\Subscription\Database\Factories\DiscountUserFactory;
use MoveOn\Subscription\Enums\DiscountDuration;
use MoveOn\Subscription\Enums\FeeType;
use MoveOn\Subscription\Models\Discount;
use MoveOn\Subscription\Models\DiscountUsageHistory;
use MoveOn\Subscription\Models\DiscountUser;
use MoveOn\Subscription\Requests\DiscountStoreRequest;
use MoveOn\Subscription\Requests\DiscountUpdateRequest;
use MoveOn\Subscription\Service\DiscountService;
use MoveOn\Subscription\Tests\TestCase;

class DiscountServiceTest extends TestCase
{
    use RefreshDatabase;

    private function getDiscountStoreRequest(array $mergeData = []): DiscountStoreRequest
    {
        $form = DiscountFactory::new()->make()->toArray();
        unset($form["gateway_coupon_id"]);

        $form["duration"]    = DiscountDuration::from($form["duration"]);
        $form["amount_type"] = FeeType::from($form["amount_type"]);
        $form                = array_merge($form, $mergeData);
        return new DiscountStoreRequest(...$form);
    }

    private function getDiscountUpdateRequest(): DiscountUpdateRequest
    {
        $request = DiscountFactory::new()->make();

        return new DiscountUpdateRequest($request->name, $request->coupon_code, $request->is_active);
    }

    public function test_discount_index()
    {
        $service = new DiscountService();
        $request = new Request();
        $list    = $service->listDiscount($request);
        $this->assertIsArray($list);
        $this->assertArrayHasKey("result", $list);
        $this->assertArrayHasKey("filters", $list);
    }


    public function test_discount_create()
    {
        $form = $this->getDiscountStoreRequest();

        $service  = new DiscountService();
        $discount = $service->createDiscount($form);

        $this->assertTrue(get_class($discount) == Discount::class);
        $this->assertDatabaseHas(Discount::class, ["id" => $discount->id]);
    }

    public function test_discount_update()
    {
        $form = $this->getDiscountStoreRequest();

        $service  = new DiscountService();
        $discount = $service->createDiscount($form);

        $service         = new DiscountService();
        $updatedDiscount = $service->updateDiscount($discount, $this->getDiscountUpdateRequest());

        $this->assertTrue(get_class($discount) == Discount::class);
        $this->assertDatabaseHas(Discount::class, ["id" => $discount->id, "name" => $updatedDiscount->name]);
    }


    public function test_discount_activate()
    {
        $form = $this->getDiscountStoreRequest(
            [
                "is_active" => false,
            ]
        );

        $service  = new DiscountService();
        $discount = $service->createDiscount($form);

        $service = new DiscountService();
        $service->activateDiscount($discount);

        $this->assertTrue(get_class($discount) == Discount::class);
        $this->assertDatabaseHas(Discount::class, ["id" => $discount->id, "is_active" => true]);
    }

    public function test_discount_deactivate()
    {
        $form = $this->getDiscountStoreRequest(
            [
                "is_active" => true,
            ]
        );

        $service  = new DiscountService();
        $discount = $service->createDiscount($form);

        $service = new DiscountService();
        $service->deactivateDiscount($discount);

        $this->assertTrue(get_class($discount) == Discount::class);
        $this->assertDatabaseHas(Discount::class, ["id" => $discount->id, "is_active" => false]);
    }

    public function test_discount_delete()
    {
        $form = $this->getDiscountStoreRequest();

        $service  = new DiscountService();
        $discount = $service->createDiscount($form);

        $service         = new DiscountService();
        $deletedDiscount = $service->deleteDiscount($discount);

        $this->assertTrue($deletedDiscount);
        $this->assertEquals(Discount::find($discount->id), null);
    }

    public function test_discount_user_index()
    {
        $service = new DiscountService();
        $request = new Request();
        $list    = $service->listDiscountUser($request);
        $this->assertIsArray($list);
        $this->assertArrayHasKey("result", $list);
        $this->assertArrayHasKey("filters", $list);
    }


    public function test_discount_user_create()
    {
        $form = $this->getDiscountStoreRequest();

        $service  = new DiscountService();
        $discount = $service->createDiscount($form);

        $request = DiscountUserFactory::new()->make()->toArray();
        unset($request["discount_id"]);
        $discountUser = $service->createDiscountUser($discount, ...$request);

        $this->assertTrue(get_class($discountUser) == DiscountUser::class);
        $this->assertDatabaseHas(DiscountUser::class, ["id" => $discountUser->id]);
    }


    public function test_discount_user_delete()
    {
        $form = $this->getDiscountStoreRequest();

        $service  = new DiscountService();
        $discount = $service->createDiscount($form);

        $request = DiscountUserFactory::new()->make()->toArray();
        unset($request["discount_id"]);
        $discountUser = $service->createDiscountUser($discount, ...$request);

        $request = DiscountUserFactory::new()->make()->toArray();
        unset($request["discount_id"]);
        $isDeleted = $service->deleteDiscountUser($discountUser);

        $this->assertTrue($isDeleted);
        $this->assertNull(DiscountUser::find($discountUser->id));
    }

    public function test_discount_user_activate()
    {
        $form = $this->getDiscountStoreRequest();

        $service  = new DiscountService();
        $discount = $service->createDiscount($form);

        $request = DiscountUserFactory::new()->make()->toArray();
        unset($request["discount_id"]);
        $discountUser = $service->createDiscountUser($discount, ...$request);

        $service     = new DiscountService();
        $isActivated = $service->activateDiscountUser($discountUser);

        $this->assertTrue($isActivated);
        $this->assertDatabaseHas(DiscountUser::class, ["id" => $discountUser->id, "is_active" => true]);
    }

    public function test_discount_user_deactivate()
    {
        $form = $this->getDiscountStoreRequest();

        $service  = new DiscountService();
        $discount = $service->createDiscount($form);

        $request = DiscountUserFactory::new()->make()->toArray();
        unset($request["discount_id"]);
        $discountUser = $service->createDiscountUser($discount, ...$request);

        $service = new DiscountService();
        $service->activateDiscountUser($discountUser);

        $service       = new DiscountService();
        $isDeActivated = $service->deactivateDiscountUser($discountUser);

        $this->assertTrue($isDeActivated);
        $this->assertDatabaseHas(DiscountUser::class, ["id" => $discountUser->id, "is_active" => false]);
    }

    public function test_discount_usage_history_index()
    {
        $service = new DiscountService();
        $request = new Request();
        $list    = $service->listDiscountUsageHistory($request);
        $this->assertIsArray($list);
        $this->assertArrayHasKey("result", $list);
        $this->assertArrayHasKey("filters", $list);
    }

    public function test_discount_usage_history_create()
    {
        $form = $this->getDiscountStoreRequest();

        $service  = new DiscountService();
        $discount = $service->createDiscount($form);

        $request = DiscountUsageHistoryFactory::new()->make()->toArray();
        unset($request["discount_id"]);
        $discountUsageHistory = $service->createDiscountUsageHistory($discount, ...$request);

        $this->assertTrue(get_class($discountUsageHistory) == DiscountUsageHistory::class);
        $this->assertDatabaseHas(DiscountUsageHistory::class, ["id" => $discountUsageHistory->id]);
    }

    public function test_discount_usage_history_update()
    {
        $form = $this->getDiscountStoreRequest();

        $service  = new DiscountService();
        $discount = $service->createDiscount($form);

        $request = DiscountUsageHistoryFactory::new()->make()->toArray();
        unset($request["discount_id"]);
        $discountUsageHistory = $service->createDiscountUsageHistory($discount, ...$request);

        $request = DiscountUsageHistoryFactory::new()->make()->toArray();
        unset($request["discount_id"]);
        $isUpdated = $service->updateDiscountUsageHistory($discountUsageHistory, $discount, ...$request);

        $this->assertTrue($isUpdated);
        $this->assertDatabaseHas(DiscountUsageHistory::class,
            [
                "id"          => $discountUsageHistory->id,
                "discount_id" => $discount->id,
                "owner_type"  => $request["owner_type"],
                "owner_id"    => $request["owner_id"],
            ]
        );
    }

}