<?php

namespace MoveOn\Subscription\Modules\Builder;

use MoveOn\Common\Traits\Makeable;
use MoveOn\Subscription\Collection\Collection;
use MoveOn\Subscription\Collection\Modules\Paypal\PaypalPlanBillingCycleCollection;
use MoveOn\Subscription\Collection\Modules\Paypal\PaypalPlanTierCollection;
use MoveOn\Subscription\Enums\FeeType;
use MoveOn\Subscription\Enums\Gateway\Paypal\PaypalPlanStatus;
use MoveOn\Subscription\Enums\Gateway\Paypal\PlanTenureType;
use MoveOn\Subscription\Enums\Gateway\Paypal\PricingModel;
use MoveOn\Subscription\Enums\IntervalUnit;
use MoveOn\Subscription\Models\Discount;
use MoveOn\Subscription\Models\Plan;
use MoveOn\Subscription\Modules\PlanDiscountModule;
use MoveOn\Subscription\Requests\Modules\Paypal\PaypalFixedPrice;
use MoveOn\Subscription\Requests\Modules\Paypal\PaypalMoney;
use MoveOn\Subscription\Requests\Modules\Paypal\PaypalPaymentPreference;
use MoveOn\Subscription\Requests\Modules\Paypal\PaypalPricingScheme;
use MoveOn\Subscription\Requests\Modules\Paypal\Plan\PaypalPlanBillingCycle;
use MoveOn\Subscription\Requests\Modules\Paypal\Plan\PaypalPlanStore;
use MoveOn\Subscription\Requests\Modules\Paypal\Plan\PaypalPlanTier;
use MoveOn\Subscription\Requests\Modules\Paypal\Plan\PlanFrequency;

class PaypalPlanRequestBuilder
{
    use Makeable;

    /**
     * @param Plan $plan
     * @param Discount $discount
     * @return PaypalPlanStore
     * @throws \Exception
     */
    public function buildPaypalPlanRequestFromPlanAndDiscount(Plan $plan, Discount $discount): PaypalPlanStore
    {
        $planTiers               = $plan->planTiers;
        $planTiersCollection     = PaypalPlanTierCollection::make();
        $currency                = $plan->currency;
        $systemUsageCharge       = 0;
        $paypalBillingCycle      = PaypalPlanBillingCycleCollection::make();
        $paypalPaymentPreference = $this->generatePaypalPaymentPreferenceRequest($currency, $systemUsageCharge);
        $product                 = $plan->product;

        foreach ($planTiers as $planTier) {
            $planTiersCollection->add(
                new PaypalPlanTier(
                    $planTier->start,
                    $planTier->end,
                    new PaypalMoney($currency, $planTier->unit_amount)
                )
            );
        }

        $paypalBillingCycle = $this->generatePaypalBillingCycleWithPlanAndDiscount(
            $plan,
            $discount,
            $paypalBillingCycle,
            $planTiersCollection
        );

        $request = [
            "name"                => "{$plan->name}-discount-{$discount->gateway_coupon_id}",
            "billing_cycles"      => $paypalBillingCycle,
            "payment_preferences" => $paypalPaymentPreference,
            "product_id"          => $product->gateway_product_id,
            "status"              => PaypalPlanStatus::ACTIVE,
            "quantity_supported"  => true,
            "description"         => $plan->description,
        ];

        return new PaypalPlanStore(...$request);
    }

    /**
     * @param string $name
     * @param int $trialPeriodDays
     * @param IntervalUnit $intervalUnit
     * @param int $intervalCount
     * @param Collection $tiers
     * @param string $currency
     * @param int $systemUsageCharge
     * @param string $gatewayProductId
     * @param $isActive
     * @return PaypalPlanStore
     * @throws \Exception
     */
    public function buildPaypalPlanRequest(
        string $name,
        int $trialPeriodDays,
        IntervalUnit $intervalUnit,
        int $intervalCount,
        Collection $tiers,
        string $currency,
        int $systemUsageCharge,
        string $gatewayProductId,
        $isActive = true
    ): PaypalPlanStore {
        $paypalBillingCycle = PaypalPlanBillingCycleCollection::make();

        if ($trialPeriodDays > 0) {
            [$paypalPricingScheme, $paypalPlanFrequency] = $this->generatePaypalTrialBillingCycleRequest(
                $trialPeriodDays
            );

            $paypalBillingCycle = $paypalBillingCycle->add(
                new PaypalPlanBillingCycle(
                    $paypalPlanFrequency,
                    1,
                    PlanTenureType::TRIAL,
                    $paypalPricingScheme,
                    1
                )
            );
        }

        $tiers = $this->generatePaypalTiersFromRequest($tiers, "USD");
        [$paypalPricingScheme, $paypalPlanFrequency] = $this->generatePaypalRegularBillingCycleRequest(
            $intervalUnit,
            $intervalCount,
            $tiers
        );

        $paypalBillingCycle = $paypalBillingCycle->add(
            new PaypalPlanBillingCycle(
                $paypalPlanFrequency,
                $trialPeriodDays > 0 ? 2 : 1,
                PlanTenureType::REGULAR,
                $paypalPricingScheme,
                1
            )
        );

        $paypalPaymentPreference = $this->generatePaypalPaymentPreferenceRequest($currency, $systemUsageCharge);

        return new PaypalPlanStore(
            $name,
            $paypalBillingCycle,
            $paypalPaymentPreference,
            $gatewayProductId,
            $isActive ? PaypalPlanStatus::ACTIVE : PaypalPlanStatus::INACTIVE,
            true,
        );
    }


    /**
     * @param Collection $tiers
     * @param string $currency
     * @return PaypalPlanTierCollection
     */
    private function generatePaypalTiersFromRequest(Collection $tiers, string $currency): PaypalPlanTierCollection
    {
        $tiers      = $tiers->items();
        $paypalTier = PaypalPlanTierCollection::make();

        foreach ($tiers as $index => $value) {
            $tiers[$index] = $paypalTier->add(
                new PaypalPlanTier($value["start"], $value["end"], new PaypalMoney($currency, $value["price"]))
            );
        }

        return $paypalTier;
    }

    /**
     * @param $currency
     * @param $systemUsageCharge
     * @return PaypalPaymentPreference
     */
    private function generatePaypalPaymentPreferenceRequest($currency, $systemUsageCharge): PaypalPaymentPreference
    {
        $paypalSetupFee = new PaypalMoney(
            $currency,
            max($systemUsageCharge, 10)
        );

        return new PaypalPaymentPreference(
            true,
            "CONTINUE",
            5,
            $paypalSetupFee
        );
    }

    /**
     * @param $trialPeriodDays
     * @return array
     * @throws \Exception
     */
    private function generatePaypalTrialBillingCycleRequest($trialPeriodDays): array
    {
        $paypalPlanFrequency = new PlanFrequency(
            IntervalUnit::PAYPAL_DAY,
            $trialPeriodDays,
        );

        $paypalPricingScheme = null;

        return [$paypalPricingScheme, $paypalPlanFrequency];
    }

    /**
     * @param IntervalUnit $intervalUnit
     * @param int $intervalCount
     * @param Collection $tiers
     * @return array
     * @throws \Exception
     */
    private function generatePaypalRegularBillingCycleRequest(
        IntervalUnit $intervalUnit,
        int $intervalCount,
        PaypalPlanTierCollection $tiers
    ): array {
        $paypalPricingScheme = new PaypalPricingScheme(
            null,
            PricingModel::TIERED,
            $tiers
        );

        $paypalPlanFrequency = new PlanFrequency(
            $intervalUnit,
            $intervalCount,
        );

        return [$paypalPricingScheme, $paypalPlanFrequency];
    }

    /**
     * @param Plan $plan
     * @param Discount $discount
     * @param IntervalUnit $intervalUnit
     * @param int $intervalCount
     * @param PaypalPlanTierCollection $tiers
     * @return array
     * @throws \Exception
     */
    private function generatePaypalRegularBillingCycleRequestWithDiscountAndPlan(
        Plan $plan,
        Discount $discount,
        IntervalUnit $intervalUnit,
        int $intervalCount,
        PaypalPlanTierCollection $tiers
    ): array {
        [$planUnitAmount, $totalDiscountAmount, $totalAmount] = PlanDiscountModule::make()->gePlanAmount($plan, $discount);

        $fixedPrice = new PaypalFixedPrice($plan->currency, (string)$totalAmount);

        $paypalPricingScheme = new PaypalPricingScheme(
            $fixedPrice,
            PricingModel::TIERED,
            $tiers
        );

        $paypalPlanFrequency = new PlanFrequency(
            $intervalUnit,
            $intervalCount,
        );

        return [$paypalPricingScheme, $paypalPlanFrequency];
    }

    /**
     * @param Plan $plan
     * @param Discount $discount
     * @param PaypalPlanBillingCycleCollection $paypalBillingCycle
     * @param PaypalPlanTierCollection $planTierCollection
     * @return PaypalPlanBillingCycleCollection
     * @throws \Exception
     */
    private function generatePaypalBillingCycleWithPlanAndDiscount(
        Plan $plan,
        Discount $discount,
        PaypalPlanBillingCycleCollection $paypalBillingCycle,
        PaypalPlanTierCollection $planTierCollection
    ): PaypalPlanBillingCycleCollection {
        $intervalCount   = $plan->interval_count;
        $trialPeriodDays = $plan->trial_period_days;
        $intervalUnit    = IntervalUnit::from($plan->interval_unit);

        if ($trialPeriodDays > 0) {
            [$paypalPricingScheme, $paypalPlanFrequency] = $this->generatePaypalTrialBillingCycleRequest(
                $trialPeriodDays
            );

            $paypalBillingCycle = $paypalBillingCycle->add(
                new PaypalPlanBillingCycle(
                    $paypalPlanFrequency,
                    1,
                    PlanTenureType::TRIAL,
                    $paypalPricingScheme,
                    1
                )
            );
        }


        [
            $paypalPricingScheme,
            $paypalPlanFrequency,
        ] = $this->generatePaypalRegularBillingCycleRequestWithDiscountAndPlan(
            $plan,
            $discount,
            $intervalUnit,
            $intervalCount,
            $planTierCollection
        );

        return $paypalBillingCycle->add(
            new PaypalPlanBillingCycle(
                $paypalPlanFrequency,
                $trialPeriodDays > 0 ? 2 : 1,
                PlanTenureType::REGULAR,
                $paypalPricingScheme,
                1
            )
        );
    }
}