<?php

namespace MoveOn\Subscription\Service;

use Illuminate\Http\Request;
use MoveOn\Common\Traits\Makeable;
use MoveOn\Subscription\Contracts\ServiceContractPaymentGateway;
use MoveOn\Subscription\Enums\FeeType;
use MoveOn\Subscription\Enums\GatewayType;
use MoveOn\Subscription\Models\PaymentGateway;
use MoveOn\Subscription\QueryFilters\PaymentGatewayFilter;
use MoveOn\Subscription\Requests\PaymentGatewayRequest;

class PaymentGatewayService implements ServiceContractPaymentGateway
{
    use Makeable;

    /**
     * List All Payment Gateways
     */
    public function list(Request $request): array
    {
        $perPage = $request->filled("per_page") ? min([$request->get("per_page"), 100]) : 10;
        $gateways = PaymentGateway::filter(PaymentGatewayFilter::class, $request)->paginate($perPage);

        return [
            "result"  => $gateways,
            "filters" => PaymentGatewayFilter::filterableDetails(),
        ];
    }

    /**
     * Retrieve Payment Gateway by ID
     */
    public function retrieve(int $id): PaymentGateway
    {
        return PaymentGateway::findOrFail($id);
    }

    /**
     * Create a Payment Gateways
     */
    public function create(PaymentGatewayRequest $form): PaymentGateway
    {
        return PaymentGateway::firstOrCreate($form->toArray());
    }

    /**
     * Update a Payment Gateways
     */
    public function update(PaymentGateway $paymentGateway, PaymentGatewayRequest $form): bool
    {
        return $paymentGateway->update($form->toArray());
    }

    /**
     * Delete a Payment Gateways
     */
    public function delete(PaymentGateway $paymentGateway): bool
    {
        return $paymentGateway->delete();
    }

    /**
     * Enable a Payment Gateways
     */
    public function enable(PaymentGateway $paymentGateway): bool
    {
        return $paymentGateway->update(
            [
                "is_active" => true,
            ]
        );
    }

    /**
     * Disable a Payment Gateways
     */
    public function disable(PaymentGateway $paymentGateway): bool
    {
        return $paymentGateway->update(
            [
                "is_active" => false,
            ]
        );
    }
}
