<?php

namespace MoveOn\Subscription\Service;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use MoveOn\Common\Traits\Makeable;
use MoveOn\Subscription\Collection\Modules\Stripe\StripeSubscriptionPaymentMethodSettingCollection;
use MoveOn\Subscription\Collection\Modules\Stripe\StripeSubscriptionStoreItemCollection;
use MoveOn\Subscription\Contracts\GatewayClientPaypal;
use MoveOn\Subscription\Contracts\GatewayClientStripe;
use MoveOn\Subscription\Contracts\ServiceContractSubscription;
use MoveOn\Subscription\Enums\Gateway\Paypal\PlanTenureType;
use MoveOn\Subscription\Enums\Gateway\Stripe\StripeSaveDefaultPaymentMethod;
use MoveOn\Subscription\Enums\Gateway\Stripe\StripeSubscriptionPaymentBehaviourEnum;
use MoveOn\Subscription\Enums\Gateway\Stripe\StripeSubscriptionPaymentMethodType;
use MoveOn\Subscription\Enums\GatewaySlug;
use MoveOn\Subscription\Enums\IntervalUnit;
use MoveOn\Subscription\Enums\UsageType;
use MoveOn\Subscription\Models\Subscription;
use MoveOn\Subscription\Models\SubscriptionItem;
use MoveOn\Subscription\Modules\Builder\PaypalPlanRequestBuilder;
use MoveOn\Subscription\Modules\Gateway\ClientPaypal;
use MoveOn\Subscription\Modules\Gateway\ClientStripe;
use MoveOn\Subscription\QueryFilters\SubscriptionFilter;
use MoveOn\Subscription\Requests\Modules\Paypal\Plan\PlanFrequency;
use MoveOn\Subscription\Requests\Modules\Paypal\Subscription\PaypalSubscriberDTORequest;
use MoveOn\Subscription\Requests\Modules\Paypal\Subscription\PaypalSubscriberNameDTORequest;
use MoveOn\Subscription\Requests\Modules\Paypal\Subscription\PaypalSubscriptionStoreDTORequest;
use MoveOn\Subscription\Requests\Modules\Stripe\StripeBillingThresholdDTORequest;
use MoveOn\Subscription\Requests\Modules\Stripe\StripeSubscriptionPaymentMethodSetting;
use MoveOn\Subscription\Requests\Modules\Stripe\StripeSubscriptionStoreDTORequest;
use MoveOn\Subscription\Requests\Modules\Stripe\StripeSubscriptionStoreItemDTORequest;
use MoveOn\Subscription\Requests\Modules\Stripe\StripeSubscriptionUpdateDTORequest;
use MoveOn\Subscription\Requests\SubscriptionCreateDTORequest;
use MoveOn\Subscription\Requests\SubscriptionUpdateDTORequest;
use MoveOn\Subscription\Requests\UsageRecordStoreDTORequest;
use Stripe\Exception\ApiErrorException;

class SubscriptionService implements ServiceContractSubscription
{
    use Makeable;

    public function listSubscription(Request $request)
    {
        $perPage = $request->filled("per_page") ? min([$request->get("per_page"), 100]) : 10;

        $subscriptions = Subscription::filter(SubscriptionFilter::class)
                                     ->with("gateway", "plan", "owner", "discount")
                                     ->paginate($perPage)
        ;

        return [
            "result"  => $subscriptions,
            "filters" => SubscriptionFilter::filterableDetails(),
        ];
    }


    /**
     *
     * @param SubscriptionCreateDTORequest $request
     * @return mixed
     * @throws \MoveOn\Subscription\Response\Gateway\Paypal\PaypalErrorResponse
     * @throws \Stripe\Exception\ApiErrorException
     * @var GatewayClientPaypal $paypal
     */
    public function createSubscription(SubscriptionCreateDTORequest $request)
    {
        return DB::transaction(function () use ($request) {
            /**
             * @var ClientStripe $stripe
             * @var ClientPaypal $paypal
             */
            $gateway           = $request->gateway;
            $form              = $request->toArray();
            $gwSubscriptionRes = [];

            if ($gateway->slug == GatewaySlug::STRIPE()) {
                $stripe = app(GatewayClientStripe::class);

                $gatewayRequest = [
                    "customer"             => $request->customer->gateway_customer_id,

                    "items"                => StripeSubscriptionStoreItemCollection::make()
                                                                                   ->add(
                                                                                       new StripeSubscriptionStoreItemDTORequest(
                                                                                           $request->plan->gateway_plan_id,
                                                                                           $request->quantity,
                                                                                           new StripeBillingThresholdDTORequest(
                                                                                               $gateway->getMeta("STRIPE_BILLING_THRESHOLD_USAGE") ?? 100
                                                                                           )
                                                                                       )
                                                                                   ),

                    "payment_behavior"     => StripeSubscriptionPaymentBehaviourEnum::DEFAULT_INCOMPLETE,
                    "payment_settings" => new StripeSubscriptionPaymentMethodSetting(
                        save_default_payment_method: StripeSaveDefaultPaymentMethod::ON_SUBSCRIPTION
                    ),
                    "trial_period_days"    => $request->plan->trial_period_days,
                    "expand"               => ["latest_invoice.payment_intent"],
                    "cancel_at_period_end" => $request->cancel_at_period_end
                ];

                if (!empty($request->discount)) {
                    $gatewayRequest["coupon"] = $request->discount->gateway_coupon_id;
                }

                $gatewaySubscription = $stripe->subscription()->createSubscription(
                    new StripeSubscriptionStoreDTORequest(...$gatewayRequest)
                );

                $form["gateway_subscription_id"] = $gatewaySubscription->id;

                $gwSubscriptionRes = [
                    "subscription_id" => $gatewaySubscription->id,
                    "client_secret"   => $gatewaySubscription?->latest_invoice?->payment_intent?->client_secret
                ];
            }

            if ($gateway->slug == GatewaySlug::PAYPAL()) {
                $paypal = app(GatewayClientPaypal::class);

                $user       = $request->customer->owner;
                $subscriber = new PaypalSubscriberDTORequest(
                    name: new PaypalSubscriberNameDTORequest(given_name: $user->name),
                    email_address: $user->email,
                );

                $gatewayRequest = [
                    "subscriber"          => $subscriber,
                    "plan_id"             => $request->plan->gateway_plan_id,
                    "quantity"            => $request->plan->default_quantity,
                    "application_context" => config("subscription.paypal.application_context"),
                ];

                if (!empty($request->discount)) {
                    $gatewayRequest["plan"] = PaypalPlanRequestBuilder::make(
                    )->buildPaypalPlanRequestFromPlanAndDiscount($request->plan, $request->discount);
                }

                $gatewaySubscription             = $paypal->subscription()->createSubscription(
                    new PaypalSubscriptionStoreDTORequest(...$gatewayRequest)
                );
                $form["gateway_subscription_id"] = $gatewaySubscription->id;

                $approve_links     = array_filter($gatewaySubscription->links, fn($item) => $item["rel"] === "approve");
                $gwSubscriptionRes = [
                    "subscription_id" => $gatewaySubscription->id,
                    "approve_link"    => count($approve_links) > 0 ? $approve_links[0]["href"] : null
                ];
            }

            if (array_key_exists("discount", $form) && !empty($form["discount"])) {
                $form["discount_id"] = $form["discount"]->id;
            }

            $form["plan_id"]    = $form["plan"]->id;
            $form["gateway_id"] = $form["gateway"]->id;
            $form["status"]     = $gatewaySubscription->status;
            unset($form["plan"], $form["discount"], $form["gateway"]);

            $subscription = Subscription::create($form);

            if ($gateway->slug == GatewaySlug::STRIPE()) {
                $subscriptionItemForm = [];
                foreach ($gatewaySubscription->items as $subscriptionItem) {
                    $subscriptionItemForm[] = [
                        "subscription_id"              => $subscription->id,
                        "plan_id"                      => $request->plan->id,
                        "gateway_subscription_item_id" => $subscriptionItem->id
                    ];
                }
                $subscription->subscriptionItems()->createMany($subscriptionItemForm);
            }

            return [
                "subscription"         => $subscription,
                "gateway_slug"         => $gateway->slug,
                "gateway_subscription" => $gwSubscriptionRes
            ];
        });
    }

    private function generatePaypalTrialBillingCycleRequest(int $trialPeriodDays)
    : array
    {
        $paypalPlanFrequency = new PlanFrequency(
            IntervalUnit::PAYPAL_DAY,
            $trialPeriodDays,
        );

        $paypalPricingScheme = null;

        return [$paypalPricingScheme, $paypalPlanFrequency];
    }

    public function cancelSubscription(Subscription $subscription)
    {
        $gateway = $subscription->gateway;

        if ($gateway->slug == GatewaySlug::STRIPE()) {
            /**
             * @var ClientStripe $client
             */
            $client = app(GatewayClientStripe::class);
        }
        if ($gateway->slug == GatewaySlug::PAYPAL()) {
            /**
             * @var ClientPaypal $client
             */
            $client = app(GatewayClientPaypal::class);
        }

        if (isset($client)) {
            return $client->subscription()->cancelSubscription($subscription->gateway_subscription_id);
        }

        throw new \Exception("Unknown gateway given");
    }

    public function subscriptionUsageUpdate(UsageRecordStoreDTORequest $request)
    {
        $subscriptionItem = SubscriptionItem::findOrFail($request->subscriptionItemId);

        /**
         * @var ClientStripe $client
         */
        $client = app(GatewayClientStripe::class);
        if (
            $client->subscriptionItem()->createUsageRecord(
                $subscriptionItem->gateway_subscription_item_id,
                $request->setQuantityAmount,
                now()->timestamp
            )) {
            return true;
        }

        return false;
    }

    /**
     * @param SubscriptionUpdateDTORequest $form
     * @return array
     * @throws ApiErrorException
     */
    public function upgradeOrDowngrade(SubscriptionUpdateDTORequest $form): array
    {
        if ($form->currentSubscription->gateway->slug === GatewaySlug::STRIPE()){
            /**
             * @var ClientStripe $client
             */
            $client = app(GatewayClientStripe::class);

            $newPlanId = $form->newPlan->gateway_plan_id;
            $currentSubscriptionId = $form->currentSubscription->gateway_subscription_id;
            $currentSubscriptionItemId = $form->currentSubscription->subscriptionItems()->latest()->first()->gateway_subscription_item_id;

            $gatewaySubscription = $client->subscription()->updateSubscription($currentSubscriptionId,
                new StripeSubscriptionUpdateDTORequest(
                    newPlanId: $newPlanId,
                    currentSubscriptionItemId: $currentSubscriptionItemId
                )
            );

            $gwSubscriptionRes = [
                "subscription_id" => $gatewaySubscription->id,
                "client_secret"   => $gatewaySubscription?->latest_invoice?->payment_intent?->client_secret
            ];


            return [
                "subscription"         => $form->currentSubscription->refresh(),
                "gateway_slug"         => GatewaySlug::STRIPE(),
                "gateway_subscription" => $gwSubscriptionRes
            ];
        }

        throw new \Exception("Given subscription does not belong to supported payment gateway");
    }


}
