<?php

namespace MoveOn\Subscription\Tests\Unit;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\Request;
use Mockery;
use Mockery\MockInterface;
use MoveOn\Subscription\Contracts\GatewayClientPaypal;
use MoveOn\Subscription\Contracts\GatewayClientStripe;
use MoveOn\Subscription\Database\Factories\ProductFactory;
use MoveOn\Subscription\Enums\GatewaySlug;
use MoveOn\Subscription\Models\PaymentGateway;
use MoveOn\Subscription\Models\Product;
use MoveOn\Subscription\Modules\Gateway\ClientPaypal;
use MoveOn\Subscription\Modules\Gateway\ClientStripe;
use MoveOn\Subscription\Requests\ProductStoreDTORequest;
use MoveOn\Subscription\Requests\ProductUpdateDTORequest;
use MoveOn\Subscription\Response\Gateway\Paypal\Product\PaypalProductResponse;
use MoveOn\Subscription\Service\ProductService;
use MoveOn\Subscription\Tests\TestCase;

class ProductServiceTest extends TestCase
{
    use RefreshDatabase;

    private function getProductCreateForm($gateway): ProductStoreDTORequest
    {
        $productForm = ProductFactory::new()->make(
            [
                "category"   => "SOFTWARE",
                "gateway_id" => $gateway->id,
            ]
        )->toArray();
        unset($productForm["gateway_product_id"], $productForm["gateway_id"]);
        $productForm["type"] = "SERVICE";
        return new ProductStoreDTORequest(...$productForm);
    }

    private function getProductUpdateForm(): ProductUpdateDTORequest
    {
        $productForm = ProductFactory::new()->make(
            [
                "category" => "SOFTWARE",
            ]
        )->toArray();

        return new ProductUpdateDTORequest(...[
            "name"        => $productForm["name"],
            "description" => $productForm["description"],
            "category"    => $productForm["category"],
        ]);
    }

    private function initStripeCreateProduct($gateway): ProductStoreDTORequest
    {
        $request = $this->getProductCreateForm($gateway);
        $this->instance(
            GatewayClientStripe::class,
            Mockery::mock(GatewayClientStripe::class, function (MockInterface $mock) {
                $id = rand(11111, 99999);
                $mock->shouldReceive('product->createProduct')->once()->andReturn(
                    json_decode(
                        <<<JSON
{
  "id": "{$id}",
  "object": "product",
  "active": true,
  "created": 1656570577,
  "default_price": null,
  "description": null,
  "images": [],
  "livemode": false,
  "metadata": {},
  "name": "Gold Special",
  "package_dimensions": null,
  "shippable": null,
  "statement_descriptor": null,
  "tax_code": null,
  "unit_label": null,
  "updated": 1656570577,
  "url": null
}
JSON
                    )
                );
            })
        );


        return $request;
    }


    private function initPaypalCreateProduct($gateway): ProductStoreDTORequest
    {
        $request = $this->getProductCreateForm($gateway);
        $this->instance(
            GatewayClientPaypal::class,
            Mockery::mock(GatewayClientPaypal::class, function (MockInterface $mock) use ($request) {
                $request       = $request->toArray();
                $request["id"] = rand(11111, 99999);
                $mock->shouldReceive('product->createProduct')->once()->andReturn(
                    new PaypalProductResponse(...$request)
                );
            })
        );


        return $request;
    }


    public function test_product_list_gives_200()
    {
        $request = new Request();
        $list    = ProductService::make()->listProduct($request);
        $this->assertIsArray($list);
        $this->assertArrayHasKey("result", $list);
        $this->assertArrayHasKey("filters", $list);
    }

    public function test_stripe_product_create_gives_200()
    {
        $gateway = PaymentGateway::where("slug", GatewaySlug::STRIPE())->first();
        $request = $this->getProductCreateForm($gateway);
        $this->instance(
            GatewayClientStripe::class,
            Mockery::mock(GatewayClientStripe::class, function (MockInterface $mock) {
                $id = rand(11111, 99999);
                $mock->shouldReceive('product->createProduct')->once()->andReturn(
                    json_decode(
                        <<<JSON
{
  "id": "{$id}",
  "object": "product",
  "active": true,
  "created": 1656570577,
  "default_price": null,
  "description": null,
  "images": [],
  "livemode": false,
  "metadata": {},
  "name": "Gold Special",
  "package_dimensions": null,
  "shippable": null,
  "statement_descriptor": null,
  "tax_code": null,
  "unit_label": null,
  "updated": 1656570577,
  "url": null
}
JSON
                    )
                );
            })
        );


        $prod = ProductService::make()->createProduct($gateway, $request);

        $this->assertTrue(get_class($prod) == Product::class);
        $this->assertDatabaseHas(Product::class, ["id" => $prod->id, "name" => $request->name]);
        $this->assertNotNull($prod->gateway_product_id);
    }

    public function test_paypal_product_create_gives_200()
    {
        $gateway = PaymentGateway::where("slug", GatewaySlug::PAYPAL())->first();
        $request = $this->initPaypalCreateProduct($gateway);
        $prod    = ProductService::make()->createProduct($gateway, $request);

        $this->assertTrue(get_class($prod) == Product::class);
        $this->assertDatabaseHas(Product::class, ["id" => $prod->id, "name" => $request->name]);
        $this->assertNotNull($prod->gateway_product_id);
    }

    public function test_stripe_product_update_gives_200()
    {
        $gateway        = PaymentGateway::where("slug", GatewaySlug::STRIPE())->first();
        $request        = $this->initStripeCreateProduct($gateway);
        $createdProduct = ProductService::make()->createProduct($gateway, $request);
        $form           = $this->getProductUpdateForm();

        $this->instance(
            GatewayClientStripe::class,
            Mockery::mock(GatewayClientStripe::class, function (MockInterface $mock) use ($createdProduct, $form) {
                $id          = $createdProduct->gateway_product_id;
                $name        = $form->name;
                $description = $form->description;
                $mock->shouldReceive('product->updateProduct')->once()->andReturn(
                    json_decode(
                        <<<JSON
{
  "id": "{$id}",
  "object": "product",
  "active": true,
  "created": 1656570577,
  "default_price": null,
  "description": "{$description}",
  "images": [],
  "livemode": false,
  "metadata": {},
  "name": "{$name}",
  "package_dimensions": null,
  "shippable": null,
  "statement_descriptor": null,
  "tax_code": null,
  "unit_label": null,
  "updated": 1656570577,
  "url": null
}
JSON
                    )
                );
            })
        );

        $updatedProduct = ProductService::make()->updateProduct($createdProduct, $form);
        $this->assertTrue(get_class($updatedProduct) == Product::class);
        $this->assertDatabaseHas(
            Product::class,
            [
                "id"                 => $createdProduct->id,
                "gateway_product_id" => $createdProduct->gateway_product_id,
                "name"               => $form->name,
                "description"        => $form->description,
            ]
        );
    }


//    public function test_paypal_product_update_gives_200()
//    {
//        $gateway        = PaymentGateway::where("slug", GatewaySlug::PAYPAL())->first();
//        $request        = $this->initPaypalCreateProduct($gateway);
//        $createdProduct = ProductService::make()->createProduct($gateway, $request);
//        $form           = $this->getProductUpdateForm();
//
//        $this->instance(
//            PaypalClient::class,
//            Mockery::mock(PaypalClient::class, function (MockInterface $mock){
//                $mock->shouldReceive('product->updateProduct')->once()->andReturn(true);
//            })
//        );
//
//        $updatedProduct = ProductService::make()->updateProduct($createdProduct, $form);
//        $this->assertTrue(get_class($updatedProduct) == Product::class);
//        $this->assertDatabaseHas(
//            Product::class,
//            [
//                "id"                 => $createdProduct->id,
//                "gateway_product_id" => $createdProduct->gateway_product_id,
//                "name"               => $form->name,
//                "description"        => $form->description,
//            ]
//        );
//    }

}