<?php

namespace MoveOn\Subscription\Service;

use Illuminate\Http\Request;
use MoveOn\Common\Traits\Makeable;
use MoveOn\Subscription\Collection\Modules\Paypal\PaypalProductUpdateCollection;
use MoveOn\Subscription\Contracts\GatewayClientPaypal;
use MoveOn\Subscription\Contracts\GatewayClientStripe;
use MoveOn\Subscription\Contracts\ServiceContractProduct;
use MoveOn\Subscription\Enums\Gateway\Paypal\PaypalProductUpdatePath;
use MoveOn\Subscription\Enums\Gateway\Paypal\PaypalUpdateOperation;
use MoveOn\Subscription\Enums\GatewaySlug;
use MoveOn\Subscription\Models\PaymentGateway;
use MoveOn\Subscription\Models\Product;
use MoveOn\Subscription\QueryFilters\ProductFilter;
use MoveOn\Subscription\Requests\Modules\Paypal\Product\PaypalProductStoreDTORequest;
use MoveOn\Subscription\Requests\Modules\Paypal\Product\PaypalProductUpdateDTORequest;
use MoveOn\Subscription\Requests\ProductStoreDTORequest;
use MoveOn\Subscription\Requests\ProductUpdateDTORequest;
use MoveOn\Subscription\Response\Gateway\Paypal\Product\PaypalProductResponse;

class ProductService implements ServiceContractProduct
{
    use Makeable;

    public function listProduct(Request $request)
    {
        $perPage   = $request->filled("per_page") ? min([$request->get("per_page"), 100]) : 10;
        $gateways = Product::filter(ProductFilter::class, $request)->paginate($perPage);

        return [
            "result"  => $gateways,
            "filters" => ProductFilter::filterableDetails(),
        ];
    }

    public function createProduct(PaymentGateway $gateway, ProductStoreDTORequest $request): Product
    {
        $form               = $request->toArray();
        $form["gateway_id"] = $gateway->id;

        unset($form["type"]);

        if ($gateway->slug == GatewaySlug::STRIPE()) {
            $stripe     = app(GatewayClientStripe::class);
            $stripeForm = [
                "name"        => $form["name"],
                "description" => $form["description"],
            ];

            if (empty($stripeForm["description"])) {
                unset($stripeForm["description"]);
            }
            $product = $stripe->product()->createProduct($stripeForm);

            $form["gateway_product_id"] = $product->id;
        }

        if ($gateway->slug == GatewaySlug::PAYPAL()) {
            $paypal     = app(GatewayClientPaypal::class);
            $paypalForm = $form;
            unset($paypalForm["gateway_id"]);

            $product = $paypal->product()->createProduct(new PaypalProductStoreDTORequest(...$paypalForm));

            $form["gateway_product_id"] = $product->id;
        }

        return Product::create($form);
    }

    public function updateProduct(Product $product, ProductUpdateDTORequest $request): Product
    {
        $form   = $request->toArray();
        $stripe = app(GatewayClientStripe::class);
        $paypal = app(GatewayClientPaypal::class);

        if (optional($product->gateway)->slug == GatewaySlug::STRIPE()) {
            $stripe->product()->updateProduct(
                $product->gateway_product_id,
                [
                    "name"        => $form["name"],
                    "description" => $form["description"],
                ]
            );
        }

        if (optional($product->gateway)->slug == GatewaySlug::PAYPAL()) {
            $paypalDetailsProduct = $paypal->product()->detailsProduct(
                $product->gateway_product_id,
            );
            if(empty($paypalDetailsProduct) || !(get_class($paypalDetailsProduct) ==  PaypalProductResponse::class)) throw new \Exception("ProductService:updateProduct: Paypal product Details not found");

            $collection = PaypalProductUpdateCollection::make()
                ->add(
                    new PaypalProductUpdateDTORequest(
                        empty($paypalDetailsProduct->category) ? PaypalUpdateOperation::ADD : PaypalUpdateOperation::REPLACE,
                        PaypalProductUpdatePath::CATEGORY,
                        $form["category"]
                    )
                )
                ->add(
                    new PaypalProductUpdateDTORequest(
                        empty($paypalDetailsProduct->description) ? PaypalUpdateOperation::ADD : PaypalUpdateOperation::REPLACE,
                        PaypalProductUpdatePath::DESCRIPTION,
                        $form["description"]
                    )
                )
            ;


            $paypal->product()->updateProduct(
                $product->gateway_product_id,
                $collection
            );

            unset($form["name"]);
        }

        $product->update($form);
        return $product;
    }
}