<?php

namespace MoveOn\Subscription\Modules\Webhook\Paypal;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use MoveOn\Subscription\Contracts\WebhookProcessorContract;
use MoveOn\Subscription\Modules\Gateway\BasePaypalGatewayManager;
use MoveOn\Subscription\Modules\Webhook\Paypal\Exceptions\UnknownEventGivenException;
use MoveOn\Subscription\Response\Gateway\Paypal\PaypalErrorResponse;

class PaypalWebhook extends BasePaypalGatewayManager implements WebhookProcessorContract
{
    private function verifyAndGetData(): array
    {
        if(!config("subscription.should_verify_request")){
            return request()->all();
        }

        /** @var String $bodyReceived */
        $bodyReceived           = file_get_contents('php://input');
        $headers                = getallheaders();
        $headers                = array_change_key_case($headers, CASE_UPPER);
        $paypalAuthAlgo         = $headers['PAYPAL-AUTH-ALGO'];
        $paypalTransmissionId   = $headers['PAYPAL-TRANSMISSION-ID'];
        $paypalCertUrl          = $headers['PAYPAL-CERT-URL'];
        $paypalWebhookId        = config("subscription.paypal.webhook.webhook_ids." . request()->route()->getName());
        $paypalTransmissionSig  = $headers['PAYPAL-TRANSMISSION-SIG'];
        $paypalTransmissionTime = $headers['PAYPAL-TRANSMISSION-TIME'];

        $verifyRequest = [
            "webhook_id"        => $paypalWebhookId,
            "transmission_id"   => $paypalTransmissionId,
            "transmission_time" => $paypalTransmissionTime,
            "cert_url"          => $paypalCertUrl,
            "auth_algo"         => $paypalAuthAlgo,
            "transmission_sig"  => $paypalTransmissionSig,
            "webhook_event"     => $bodyReceived,
        ];

        if (empty($paypalWebhookId)) {
            throw new \Exception("Paypal webhook id got empty", 500);
        }

        $response = Http::withHeaders(
            [
                "Content-Type" => "application/json",
            ]
        )
            ->withToken($this->getToken()->access_token)
            ->post(
                config("subscription.paypal.base_url") . "/billing/plans",
                $verifyRequest
            );

        if ($this->isSuccessResponse($response->status())) {
            if ($response->json()["verification_status"] == "SUCCESS") {
                return request()->all();
            }

            Log::critical("Webhook Verification failed", $response->json());
            throw new PaypalErrorResponse(get_class($this) . ":verifyPaypalWebhook", $response->json());
        }

        throw new PaypalErrorResponse(
            get_class($this) . ":verifyPaypalWebhook", $response->json(), $response->status()
        );
    }

    private function dispatchEvent(array $payload)
    {
        $events = $this->getEvents();

        $eventName = $payload["event_type"];

        if (!array_key_exists($eventName, $events)) {
            throw new \Exception(get_class($this) . " Unknown Event given. payload: " . json_encode($payload), 500);
        }
        if (!empty($events[$eventName])) {
            event(new $events[$eventName]($payload));
        }
    }

    protected function getEvents(): array
    {
        return [
            "PAYMENT.SALE.COMPLETED"              => config("subscription.paypal.events.webhook.payment_sale_completed"),
            "PAYMENT.SALE.REVERSED"               => config("subscription.paypal.events.webhook.payment_sale_reversed"),
            "BILLING.PLAN.ACTIVATED"              => config("subscription.paypal.events.webhook.billing_plan_activated"),
            "BILLING.PLAN.PRICING"                => config("subscription.paypal.events.webhook.billing_plan_pricing_change_activated"),
            "BILLING.PLAN.DEACTIVATED"            => config("subscription.paypal.events.webhook.billing_plan_deactivated"),
            "BILLING.SUBSCRIPTION.ACTIVATED"      => config("subscription.paypal.events.webhook.billing_subscription_activated"),
            "BILLING.SUBSCRIPTION.UPDATED"        => config("subscription.paypal.events.webhook.billing_subscription_updated"),
            "BILLING.SUBSCRIPTION.EXPIRED"        => config("subscription.paypal.events.webhook.billing_subscription_expired"),
            "BILLING.SUBSCRIPTION.CANCELLED"      => config("subscription.paypal.events.webhook.billing_subscription_cancelled"),
            "BILLING.SUBSCRIPTION.SUSPENDED"      => config("subscription.paypal.events.webhook.billing_subscription_suspended"),
            "BILLING.SUBSCRIPTION.PAYMENT.FAILED" => config("subscription.paypal.events.webhook.billing_subscription_payment_failed"),
        ];
    }

    public function process(): void
    {
        // Send request to paypal server to verify the incoming data
        $webHookData = $this->verifyAndGetData();

        // Dispatch event
        $this->dispatchEvent($webHookData);

        http_response_code(200);
    }

}