<?php

namespace MoveOn\Subscription\Service;

use MoveOn\Common\Traits\Makeable;
use MoveOn\Subscription\Contracts\GatewayClientStripe;
use MoveOn\Subscription\Contracts\ServiceContractInvoice;
use MoveOn\Subscription\Enums\Gateway\Stripe\StripeInvoiceStatusTypeEnum;
use MoveOn\Subscription\Models\Invoice;
use MoveOn\Subscription\Models\StripeInvoice;
use MoveOn\Subscription\Modules\Gateway\ClientStripe;
use MoveOn\Subscription\Requests\InvoiceUpdateDTORequest;
use MoveOn\Subscription\Requests\Modules\Stripe\StripeInvoiceUpdateDTORequest;

class InvoiceService implements ServiceContractInvoice
{
    use Makeable;

    /**
     * @param string $invoiceId
     * @return \Stripe\Invoice
     * @throws \Exception
     */
    public function detail(string $invoiceId): \Stripe\Invoice
    {
        $invoice = Invoice::with('detail')->findOrFail($invoiceId);

        if($invoice->detail instanceof StripeInvoice){
            /** @var ClientStripe $client */
            $client = app(GatewayClientStripe::class);
            return $client->invoice()->retrieve($invoice->detail->gateway_invoice_id);
        }

        throw new \Exception("Invoice not supported for this gateway");
    }

    /**
     * @param string $invoiceId
     * @param InvoiceUpdateDTORequest $request
     * @return \Stripe\Invoice
     * @throws \Exception
     */
    public function update(string $invoiceId, InvoiceUpdateDTORequest $request)
    {
        $invoice = Invoice::with('detail')->findOrFail($invoiceId);

        if($invoice->detail instanceof StripeInvoice){
            /** @var ClientStripe $client */
            $client = app(GatewayClientStripe::class);
            $gatewayInvoice = $client->invoice()->retrieve($invoice->detail->gateway_invoice_id);
            if($gatewayInvoice->status == StripeInvoiceStatusTypeEnum::PAID()){
                return $gatewayInvoice;
            }

            return $client->invoice()->update($invoice->detail->gateway_invoice_id, new StripeInvoiceUpdateDTORequest(
                auto_advance: true,
            ));
        }

        throw new \Exception("Invoice not supported for this gateway");
    }
}